
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Simulated data for tourist arrivals (thousands) over months
months = np.linspace(1, 12, 12)  # 12 months
tourist_arrivals = np.array([20, 25, 30, 45, 60, 70, 80, 90, 85, 60, 40, 30])
tourist_arrivals_pred = tourist_arrivals + np.random.normal(0, 5, len(tourist_arrivals))

# Simulated data for hotel occupancy rate (%) over months
hotel_occupancy = np.array([50, 55, 60, 70, 85, 90, 95, 85, 75, 65, 55, 50])
hotel_occupancy_pred = hotel_occupancy + np.random.normal(0, 5, len(hotel_occupancy))

# Extracted variables
tourist_arrivals_label = "Actual Tourist Arrivals"
tourist_arrivals_pred_label = "Predicted Tourist Arrivals"
hotel_occupancy_label = "Actual Hotel Occupancy"
hotel_occupancy_pred_label = "Predicted Hotel Occupancy"

tourist_arrivals_ylabel = "Tourist Arrivals (thousands)"
hotel_occupancy_xlabel = "Months"
hotel_occupancy_ylabel = "Hotel Occupancy Rate (%)"
legend_labels = ["Actual", "Predicted"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
tourist_arrivals_ylim = [0, 100]
tourist_arrivals_xlim = [1, 12]
tourist_arrivals_yticks = [0, 20, 40, 60, 80, 100]
tourist_arrivals_xticks = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]
hotel_occupancy_ylim = [40, 100]
hotel_occupancy_xlim = [1, 12]
hotel_occupancy_yticks = [40, 50, 60, 70, 80, 90, 100]
hotel_occupancy_xticks = tourist_arrivals_xticks

legend_loc = "lower center"
legend_bbox_to_anchor = (0.5, -0.2)
legend_ncol = 2
legend_frameon = False

# Create figure and axes
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(12, 12))

# Plot Tourist Arrivals
ax1.plot(months, tourist_arrivals, "o-", color="#2ca02c", label=tourist_arrivals_label)
ax1.fill_between(months, tourist_arrivals, color="#b2df8a", alpha=0.5)
ax1.plot(months, tourist_arrivals_pred, "s-", color="#8c564b", label=tourist_arrivals_pred_label)
ax1.fill_between(months, tourist_arrivals_pred, color="#c49c94", alpha=0.5)
ax1.set_ylim(tourist_arrivals_ylim)
ax1.set_xlim(tourist_arrivals_xlim)
ax1.set_yticks(tourist_arrivals_yticks)
ax1.set_xticks(tourist_arrivals_xticks)
ax1.set_ylabel(tourist_arrivals_ylabel)
ax1.tick_params(axis="both", which="both", length=0)

# Plot Hotel Occupancy
ax2.plot(months, hotel_occupancy, "o-", color="#2ca02c", label=hotel_occupancy_label)
ax2.fill_between(months, hotel_occupancy, color="#b2df8a", alpha=0.5)
ax2.plot(
    months,
    hotel_occupancy_pred,
    "s-",
    color="#8c564b",
    label=hotel_occupancy_pred_label,
)
ax2.fill_between(months, hotel_occupancy_pred, color="#c49c94", alpha=0.5)
ax2.set_ylim(hotel_occupancy_ylim)
ax2.set_xlim(hotel_occupancy_xlim)
ax2.set_yticks(hotel_occupancy_yticks)
ax2.set_xticks(hotel_occupancy_xticks)
ax2.set_xlabel(hotel_occupancy_xlabel)
ax2.set_ylabel(hotel_occupancy_ylabel)
ax2.tick_params(axis="both", which="both", length=0)

# Create custom legend
green_line = mlines.Line2D(
    [], [], color="#2ca02c", marker="o", markersize=6, label=legend_labels[0]
)
brown_line = mlines.Line2D(
    [], [], color="#8c564b", marker="s", markersize=6, label=legend_labels[1]
)
plt.legend(
    handles=[green_line, brown_line],
    loc=legend_loc,
    bbox_to_anchor=legend_bbox_to_anchor,
    ncol=legend_ncol,
    frameon=legend_frameon,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("area_44.pdf", bbox_inches="tight")
