# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Set the seed for reproducibility
import numpy as np

np.random.seed(42)

# Data
years = np.arange(2010, 2020)
sales_growth = np.linspace(50, 150, len(years)) + np.random.normal(scale=5, size=len(years))
market_share = np.linspace(20, 40, len(years)) + np.random.normal(scale=1, size=len(years))
customer_satisfaction = np.linspace(70, 90, len(years)) + np.random.normal(scale=2, size=len(years))

# Variables for plot configuration
sales_growth_label = "Annual Sales Growth (%)"
market_share_label = "Market Share (%)"
customer_satisfaction_label = "Customer Satisfaction (index)"
xlabel_text = "Years"
ylabel_text = "Business Performance Metrics"
title_text = "Business Performance Trends (2010-2020)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
ylim_values = (0, 300)
xticks_values = years
yticks_values = [0, 50, 100, 150, 200, 250, 300]
legend_location = "upper left"
legend_fontsize = 12

# Plot
fig, ax = plt.subplots(figsize=(10, 6))  # Adjusted for better aspect ratio

# Stack the data
stack_data = np.vstack((sales_growth, market_share, customer_satisfaction))

# Colors for each stack
colors = ["#1f78b4", "#a6cee3", "#b2df8a"]

# Area plot
ax.stackplot(
    years,
    stack_data,
    labels=[sales_growth_label, market_share_label, customer_satisfaction_label],
    colors=colors,
    alpha=0.8,
)

# Remove top and right spines
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)

# Setting the x-axis and y-axis limits and ticks
ax.set_ylim(*ylim_values)
ax.set_xticks(xticks_values)
ax.set_yticks(yticks_values)

# Adding labels and title
ax.set_xlabel(xlabel_text, fontsize=14)
ax.set_ylabel(ylabel_text, fontsize=14)
ax.set_title(title_text, fontsize=16, y=1.05)

# Custom legend
ax.legend(loc=legend_location, fontsize=legend_fontsize, frameon=False)

# Grid
ax.grid(True, linestyle="--", alpha=0.5, which="both")

# ===================
# Part 4: Saving Output
# ===================
# Adjusting layout to reduce white space
plt.tight_layout()
plt.savefig("area_68.pdf", bbox_inches="tight")
