
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Simulated data for media domain
time_step = np.linspace(0, 24, 48)
number_of_likes = 500 + 200 * np.sin(np.pi * time_step / 12) + np.random.normal(0, 50, 48)
number_of_comments = 80 + np.cumsum(np.random.normal(0, 10, 48))

# Extracted variables for labels and settings
likes_label = "Number of Likes"
comments_label = "Number of Comments"
likes_ylabel = "Number of Likes"
comments_xlabel = "Time (hours)"
comments_ylabel = "Number of Comments"
legend_labels = ["Number of Likes", "Number of Comments"]

# Plot configuration parameters
likes_ylim = [min(number_of_likes) - 50, max(number_of_likes) + 50]
likes_xlim = [0, 24]
comments_ylim = [min(number_of_comments) - 20, max(number_of_comments) + 20]
comments_xlim = [0, 24]
likes_yticks = np.arange(min(number_of_likes) // 100 * 100, max(number_of_likes) // 100 * 100 + 200, 100)
comments_yticks = np.arange(min(number_of_comments) // 20 * 20, max(number_of_comments) // 20 * 20 + 40, 20)
time_xticks = np.arange(0, 25, 4)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

# Colors for plotting
likes_color = "#1f77b4"
comments_color = "#ff7f0e"
likes_fill_color = "#aec7e8"
comments_fill_color = "#ffbb78"

# Create figure and axes
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 10))

# Plot Number of Likes
ax1.plot(time_step, number_of_likes, "o-", color=likes_color, label=likes_label)
ax1.fill_between(time_step, number_of_likes, color=likes_fill_color, alpha=0.3)
ax1.set_ylim(likes_ylim)
ax1.set_xlim(likes_xlim)
ax1.set_yticks(likes_yticks)
ax1.set_xticks(time_xticks)
ax1.set_ylabel(likes_ylabel)
ax1.tick_params(axis="both", which="both", length=0)
ax1.grid(True)

# Plot Number of Comments
ax2.plot(time_step, number_of_comments, "s--", color=comments_color, label=comments_label)
ax2.fill_between(time_step, number_of_comments, color=comments_fill_color, alpha=0.3)
ax2.set_ylim(comments_ylim)
ax2.set_xlim(comments_xlim)
ax2.set_yticks(comments_yticks)
ax2.set_xticks(time_xticks)
ax2.set_xlabel(comments_xlabel)
ax2.set_ylabel(comments_ylabel)
ax2.tick_params(axis="both", which="both", length=0)
ax2.grid(True)

# Create custom legend
blue_line = mlines.Line2D(
    [], [], color=likes_color, marker="o", markersize=6, label=legend_labels[0]
)
orange_line = mlines.Line2D(
    [], [], color=comments_color, marker="s", markersize=6, label=legend_labels[1]
)
plt.legend(
    handles=[blue_line, orange_line],
    loc="lower center",
    bbox_to_anchor=(0.5, -0.2),
    ncol=2,
    frameon=False,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("area_80.pdf", bbox_inches="tight")
