
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)
# Data
products = ["Product A", "Product B", "Product C"]
sales_performance = {
    "Q1 2020": [120, 150, 130],
    "Q2 2020": [140, 160, 155],
    "Q3 2020": [180, 170, 150],
    "Q4 2020": [200, 210, 190],
    "Q1 2021": [220, 230, 205],
    "Q2 2021": [250, 240, 220],
}

# Plot labels and types
ylabel_text = "Sales Performance (in units)"
xlabel_text = "Products"
title_text = "Quarterly Sales Performance Comparison"
legend_title = "Quarters"

# Plot limits and ticks
ylim_values = (50, 300)  # sales values range from 50 to 300
yticks_values = [50, 100, 150, 200, 250, 300]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
bar_label_fontsize = 10
ylabel_fontsize = 12

# Colors for the bars
colors = ["#4c72b0", "#55a868", "#c44e52", "#8172b3", "#ccb974", "#64b5cd"]

# Setup the figure and axes
fig, ax = plt.subplots(figsize=(12, 4))

# Bar width
bar_width = 0.1

# Positions of the bars on the x-axis
r = np.arange(len(products))

# Create bars for each training condition
for i, (quarter, values) in enumerate(sales_performance.items()):
    ax.bar(
        r + i * bar_width,
        values,
        color=colors[i],
        width=bar_width,
        edgecolor="white",
        label=quarter,
    )

# Add labels on top of the bars
for i, (quarter, values) in enumerate(sales_performance.items()):
    for j, value in enumerate(values):
        ax.text(
            j + i * bar_width,
            value + 2,
            f"{value:.0f}",
            ha="center",
            va="bottom",
            fontsize=bar_label_fontsize,
        )

# General layout
ax.set_ylabel(ylabel_text, fontsize=ylabel_fontsize)
ax.set_xticks(r + bar_width * (len(sales_performance) - 1) / 2)
ax.set_xticklabels(products)
ax.set_ylim(*ylim_values)
ax.set_yticks(yticks_values)
ax.legend(
    title=legend_title,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.2),
    frameon=False,
    ncol=7,
)
ax.set_facecolor("#f0f0f5")
ax.yaxis.grid(True, color="white")
ax.set_axisbelow(True)

# Remove spines
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(False)
ax.spines["left"].set_visible(False)

plt.tick_params(axis="both", which="both", length=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(12, 4)
plt.tick_params(axis="both", which="both", length=0)
plt.tight_layout()
plt.savefig("bar_189.pdf", bbox_inches="tight")
