
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
companies = ["Apple", "Microsoft", "Amazon"]
market_values = {
    "1980": [0.1, 0.3, 0.05],
    "1990": [0.5, 0.4, 0.1],
    "2000": [0.6, 0.5, 0.2],
    "2010": [0.9, 0.6, 0.5],
    "2020": [2.0, 1.6, 1.8],
}

# Plot labels and types
ylabel_text = "Market Value in Trillions of Dollars"
xlabel_text = "Technology Companies"
title_text = "Market Value of Technology Companies Over Decades"
legend_title = "Decades"

# Plot limits and ticks
ylim_values = (0, 2.5)
yticks_values = [0, 0.5, 1.0, 1.5, 2.0, 2.5]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
bar_label_fontsize = 10
ylabel_fontsize = 12

# Colors for the bars
colors = ["#7851a9", "#e65c00", "#9acd32", "#4682b4", "#ffcc00"]

# Setup the figure and axes
fig, ax = plt.subplots(figsize=(12, 4))

# Bar width
bar_width = 0.1

# Positions of the bars on the x-axis
r = np.arange(len(companies))

# Create bars for each decade
for i, (decade, values) in enumerate(market_values.items()):
    ax.bar(
        r + i * bar_width,
        values,
        color=colors[i % len(colors)],
        width=bar_width,
        edgecolor="white",
        label=decade,
    )

# Add labels on top of the bars
for i, (decade, values) in enumerate(market_values.items()):
    for j, value in enumerate(values):
        ax.text(
            j + i * bar_width,
            value + 0.02,
            str(value),
            ha="center",
            va="bottom",
            fontsize=bar_label_fontsize,
        )

# General layout
ax.set_ylabel(ylabel_text, fontsize=ylabel_fontsize)
ax.set_xticks(r + bar_width * (len(market_values) - 1) / 2)
ax.set_xticklabels(companies)
ax.set_ylim(*ylim_values)
ax.set_yticks(yticks_values)
ax.legend(
    title=legend_title,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.2),
    frameon=False,
    ncol=5,
)
ax.set_facecolor("#f9f9f9")
ax.yaxis.grid(True, color="white")
ax.set_axisbelow(True)

# Remove spines
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(False)
ax.spines["left"].set_visible(False)

plt.tick_params(axis="both", which="both", length=0)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(12, 4)
plt.tick_params(axis="both", which="both", length=0)
plt.tight_layout()
plt.savefig("bar_190.pdf", bbox_inches="tight")
