

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data
issues = [
    "Revenue Growth",
    "Customer Satisfaction",
    "Market Share",
    "Employee Productivity",
    "Operational Efficiency",
    "Cost Reduction",
    "Brand Value",
    "Product Innovation",
    "Customer Retention",
    "Supply Chain",
    "Net Profit Margin",
    "Market Expansion",
    "Digital Transformation",
    "Return on Investment",
    "Cash Flow"
]
delta_opinion = [
    10,
    8,
    -6,
    5,
    -8,
    -3,
    12,
    4,
    2,
    -7,
    6,
    1,
    7,
    9,
    3
]

# Extracted variables
xlim_values = (-1, len(issues))
ylim_values = (-10, 15)
xlabel_text = "Business Indicators"
ylabel_text = "Change in Performance (%)"
title_text = "Performance Change Across Business Indicators"
legend_labels = ["Negative Change", "Neutral Change", "Positive Change"]
yticks_values = np.arange(-10, 16, 5)

# Text labels for annotations
annotation_text_below = [issue for delta, issue in zip(delta_opinion, issues) if delta < 0]
annotation_text_above = [issue for delta, issue in zip(delta_opinion, issues) if delta >= 0]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors based on delta opinion
colors = [
    "#d9534f" if x < 0 else "#f0ad4e" if 0 <= x <= 1 else "#5cb85c"
    for x in delta_opinion
]

# Plot
fig, ax = plt.subplots(figsize=(10, 6))
bars = ax.bar(issues, delta_opinion, color=colors)

# Labels and Title
ax.set_xlim(*xlim_values)
ax.set_xticks(range(len(issues)))
ax.set_xticklabels(issues, rotation=45, ha="right", fontsize=9)
ax.set_ylim(*ylim_values)
ax.set_yticks(yticks_values)
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Add text labels
for bar, delta in zip(bars, delta_opinion):
    y = bar.get_height()
    if y < 0:
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            y - 0.5,
            f"{y}%",
            ha="center",
            va="top",
            fontsize=8,
            color="white"
        )
    else:
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            y + 0.5,
            f"{y}%",
            ha="center",
            va="bottom",
            fontsize=8
        )

# Legend
negative_patch = plt.Rectangle((0, 0), 1, 1, fc="#d9534f", edgecolor="none")
neutral_patch = plt.Rectangle((0, 0), 1, 1, fc="#f0ad4e", edgecolor="none")
positive_patch = plt.Rectangle((0, 0), 1, 1, fc="#5cb85c", edgecolor="none")
ax.legend(
    [negative_patch, neutral_patch, positive_patch],
    legend_labels,
    loc="upper left",
    bbox_to_anchor=(1, 1),
    frameon=False
)

plt.tick_params(axis="both", which="both", length=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout
plt.tight_layout()
plt.savefig("bar_204.pdf", bbox_inches="tight")

