
# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Define the data and category names as provided by the user
category_names = [
    "Strongly disagree",
    "Disagree",
    "Neither agree nor disagree",
    "Agree",
    "Strongly agree",
]
results = {
    "Public Transit Accessibility": [20, 25, 15, 25, 15],
    "Road Infrastructure": [30, 20, 10, 25, 15],
    "Bike Lanes": [15, 20, 25, 20, 20],
    "Traffic Congestion": [25, 15, 20, 20, 20],
    "Car Emissions Policies": [10, 15, 20, 30, 25],
}
ylim = [-70, 70]
yticks = np.arange(-70, 71, 20)
xlabel = "Responses"
title = "Public Opinion on Transportation Policies"
axhline = 0

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
def create_vertical_bar_chart(results, category_names):
    fig, ax = plt.subplots(figsize=(10, 6))

    labels = list(results.keys())
    data = np.array(list(results.values()))
    data_cum = data.cumsum(axis=1)
    middle_index = data.shape[1] // 2
    offsets = data[:, range(middle_index)].sum(axis=1) + data[:, middle_index] / 2

    # Color Mapping
    category_colors = ['#1f77b4', '#aec7e8', '#ff7f0e', '#ffbb78', '#2ca02c']

    # Plot Bars
    for i, (colname, color) in enumerate(zip(category_names, category_colors)):
        heights = data[:, i]
        bottoms = data_cum[:, i] - heights - offsets
        rects = ax.bar(
            labels,
            heights,
            bottom=bottoms,
            width=0.5,
            label=colname,
            color=color,
            edgecolor="black",
        )
        for j, (bottom, height) in enumerate(zip(bottoms, heights)):
            # Calculate the center position of each bar segment for the text
            text_x = bottom + height / 2
            text_y = j  # y-coordinate is based on the bar's index (j)
            ax.text(
                text_y,
                text_x,
                f"{abs(height):.1f}%",
                va="center",
                ha="center",
                color="black",
                fontsize=8,
            )
    # Add Zero Reference Line
    ax.axhline(axhline, linestyle="--", color="black", linewidth=1)

    # X Axis
    ax.set_ylim(ylim)
    ax.set_yticks(yticks)
    ax.yaxis.set_major_formatter(lambda y, pos: str(abs(int(y))))

    # Y Axis
    ax.set_xlabel(xlabel)
    ax.set_title(title)

    # Remove spines
    ax.spines["top"].set_visible(False)
    ax.spines["right"].set_visible(False)

    # Legend
    ax.legend(
        ncol=len(category_names),
        bbox_to_anchor=(0.5, 1.15),
        loc="upper center",
        frameon=False,
    )

    return fig, ax


fig, ax = create_vertical_bar_chart(results, category_names)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("bar_316.pdf", bbox_inches="tight")
