

# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Define data and category names for energy consumption satisfaction survey
category_names = [
    "Very Unsatisfied",
    "Unsatisfied",
    "Neutral",
    "Satisfied",
    "Very Satisfied",
]
results = {
    "Solar": [10, 14, 20, 35, 21],
    "Wind": [15, 18, 23, 28, 16],
    "Hydro": [12, 17, 22, 30, 19],
    "Nuclear": [20, 25, 24, 18, 13],
    "Coal": [25, 30, 28, 10, 7],
}
ylim = [-100, 100]
yticks = np.arange(-100, 101, 25)
xlabel = "Responses"
axhline = 0
title = "Satisfaction with Various Energy Sources"
suptitle = "Survey on Energy Consumption Satisfaction"
savefile = "bar_317_1.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
def create_vertical_bar_chart(results, category_names):
    fig, ax = plt.subplots(figsize=(10, 6))

    labels = list(results.keys())
    data = np.array(list(results.values()))
    data_cum = data.cumsum(axis=1)
    middle_index = data.shape[1] // 2
    offsets = data[:, range(middle_index)].sum(axis=1) + data[:, middle_index] / 2

    # Color Mapping
    category_colors = plt.get_cmap("tab20c")(np.linspace(0.15, 0.85, data.shape[1]))

    # Plot Bars
    for i, (colname, color) in enumerate(zip(category_names, category_colors)):
        heights = data[:, i]
        bottoms = data_cum[:, i] - heights - offsets
        rects = ax.bar(
            labels,
            heights,
            bottom=bottoms,
            width=0.5,
            label=colname,
            color=color,
            edgecolor="black",
        )
        for j, (bottom, height) in enumerate(zip(bottoms, heights)):
            # Calculate the center position of each bar segment for the text
            text_x = bottom + height / 2
            text_y = j  # y-coordinate is based on the bar's index (j)
            ax.text(
                text_y,
                text_x,
                f"{abs(height):.1f}%",
                va="center",
                ha="center",
                color="black",
                fontsize=8,
            )
    # Add Zero Reference Line
    ax.axhline(axhline, linestyle="--", color="black", alpha=0.25)

    # X Axis
    ax.set_ylim(ylim)
    ax.set_yticks(yticks)
    ax.yaxis.set_major_formatter(lambda y, pos: str(abs(int(y))))

    # Y Axis
    ax.set_xlabel(xlabel)

    # Remove spines
    ax.spines["top"].set_visible(False)
    ax.spines["right"].set_visible(False)

    # Titles and Legends
    fig.suptitle(suptitle, fontsize=14)
    ax.set_title(title, fontsize=12, pad=20)
    ax.legend(
        ncol=len(category_names) // 2,
        bbox_to_anchor=(0.5, 1),
        loc="upper center",
        frameon=False,
    )

    return fig, ax


fig, ax = create_vertical_bar_chart(results, category_names)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("bar_317.pdf", bbox_inches="tight")

