
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Patch

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Sample data for demonstration purposes: Energy Consumption across Sources
data_solar = [
    np.random.normal(400, 50, 20),  # Urban
    np.random.normal(350, 45, 20),  # Suburban
    np.random.normal(300, 40, 20),  # Rural
]
data_wind = [
    np.random.normal(200, 30, 20),  # Urban
    np.random.normal(180, 25, 20),  # Suburban
    np.random.normal(150, 20, 20),  # Rural
]
data_hydro = [
    np.random.normal(500, 60, 20),  # Urban
    np.random.normal(450, 55, 20),  # Suburban
    np.random.normal(400, 50, 20),  # Rural
]

# Combine data
data = [data_solar, data_wind, data_hydro]
xticklabels = ["Solar", "Wind", "Hydro"]
ylabel = "Energy Consumption (kWh)"
xticks = [1, 3, 5]
ylim = [100, 700]
labels = ["Urban", "Suburban", "Rural"]
title = "Energy Consumption Across Different Sources"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specific size
fig, ax = plt.subplots(figsize=(6, 5))  # Slightly larger to ensure clarity

# Plot boxplots with different colors and hatch patterns
colors = ["#66c2a5", "#fc8d62", "#8da0cb"]  # Green, Orange, Blue
hatches = ["//", "\\\\", "oo"]  # Different hatches for distinction

for i, d in enumerate(data):
    bp = ax.boxplot(
        d,
        positions=np.array(range(len(d))) * 2 + (i + 1) * 0.5,
        widths=0.5,
        patch_artist=True,
    )
    for patch in bp["boxes"]:
        patch.set_facecolor(colors[i])
        patch.set_hatch(hatches[i])
        patch.set_edgecolor("black")  # Set edge color to black
    for flier in bp["fliers"]:
        flier.set(
            marker="d", color="black", markerfacecolor="black", markersize=4, alpha=0.5
        )

# Customizing the plot to match the given picture
ax.set_xticks(xticks)
ax.set_xticklabels(xticklabels)
ax.set_ylabel(ylabel)
ax.set_ylim(ylim)
ax.set_title(title)

# Adding legend manually to match the picture
legend_elements = [
    Patch(facecolor=colors[0], hatch=hatches[0], label=labels[0], edgecolor="black"),
    Patch(facecolor=colors[1], hatch=hatches[1], label=labels[1], edgecolor="black"),
    Patch(facecolor=colors[2], hatch=hatches[2], label=labels[2], edgecolor="black"),
]
ax.legend(handles=legend_elements, loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("box_48.pdf", bbox_inches="tight")
