# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Setting the random seed for reproducibility
np.random.seed(0)

# Sample data simulating case win rates
data1 = [
    np.random.normal(70, 10, 100),  # Civil Cases
    np.random.normal(50, 15, 100),  # Criminal Cases
    np.random.normal(60, 12, 100),  # Corporate Cases
]

data2 = [
    np.random.normal(72, 8, 100),  # Civil Cases
    np.random.normal(55, 14, 100),  # Criminal Cases
    np.random.normal(65, 10, 100),  # Corporate Cases
]

titles = ["Law Firm A Case Win Rates", "Law Firm B Case Win Rates"]
ylabels = ["Win Rate (%)", "Win Rate (%)"]
xticklabels = ["Civil Cases", "Criminal Cases", "Corporate Cases"]
yticks = [np.arange(0, 101, 10), np.arange(0, 101, 10)]

medianprops = dict(color="black", linestyle="--", linewidth=1.5)
boxprops = dict(color="blue", facecolor="lightblue")

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with custom size
fig, axs = plt.subplots(1, 2, figsize=(12, 6))
# Additional style parameters
color_scheme = ["#1f77b4", "#2ca02c", "#ff7f0e"]  # Blue, Green, Orange

# Create the boxplots
for i, data in enumerate([data1, data2]):
    bp = axs[i].boxplot(data, medianprops=medianprops)

    # Customize boxplot colors
    for box in bp["boxes"]:
        box.set(color=color_scheme[i % len(color_scheme)], linewidth=1.5)
    for whisker in bp["whiskers"]:
        whisker.set(color=color_scheme[i % len(color_scheme)], linestyle="--")
    for cap in bp["caps"]:
        cap.set(color=color_scheme[i % len(color_scheme)])
    for flier in bp["fliers"]:
        flier.set(marker="o", color=color_scheme[i % len(color_scheme)], alpha=0.5)
    for median in bp["medians"]:
        median.set(color="black", linewidth=1.5)

    # Set titles and labels
    axs[i].set_title(titles[i])
    axs[i].set_ylabel(ylabels[i])
    axs[i].set_xticklabels(xticklabels, rotation=45)
    axs[i].set_yticks(yticks[i])
    axs[i].grid(True, linestyle="--", linewidth=0.7, alpha=0.7)

# Adjust layout for better spacing
plt.subplots_adjust(left=0.08, right=0.92, top=0.9, bottom=0.1, wspace=0.4)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("box_50.pdf", bbox_inches="tight")
