
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Patch

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Simulate data for economic trends distribution
years = np.linspace(0, 10, 500)  # Simulate a timeline from 0 to 10 decades
regions = np.linspace(0, 10, 500)  # Simulate different regions from 0 to 10
Years, Regions = np.meshgrid(years, regions)

# Define GDP growth density functions for different periods
def gdp_density(years, regions, year_mean, region_mean, year_std, region_std):
    return np.exp(-(((years - year_mean) ** 2) / (2 * year_std**2) + ((regions - region_mean) ** 2) / (2 * region_std**2)))

# Generate data
GDP1 = gdp_density(Years, Regions, 2, 5, 1.5, 1.0)  # Early Period
GDP2 = gdp_density(Years, Regions, 5, 7, 1.0, 1.5)  # Middle Period
GDP3 = gdp_density(Years, Regions, 8, 3, 1.2, 1.2)  # Recent Period

# Chart titles and labels
title = "Density of GDP Growth Over Decades"
xlabel = "Decades"
ylabel = "Regions"
labels = ["Early Growth", "Middle Growth", "Recent Growth"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
legend_fontsize = 12
title_fontsize = 16
label_fontsize = 14

# Plotting
plt.figure(figsize=(12, 8))
contour1 = plt.contourf(Years, Regions, GDP1, cmap="Greens", alpha=0.6)
contour2 = plt.contourf(Years, Regions, GDP2, cmap="Oranges", alpha=0.4)
contour3 = plt.contourf(Years, Regions, GDP3, cmap="Purples", alpha=0.3)

# Title and labels
plt.title(title, fontsize=title_fontsize)
plt.xlabel(xlabel, fontsize=label_fontsize)
plt.ylabel(ylabel, fontsize=label_fontsize)

# Create legend with color patches
legend_patches = [
    Patch(color="green", label=labels[0]),
    Patch(color="orange", label=labels[1]),
    Patch(color="purple", label=labels[2]),
]
plt.legend(handles=legend_patches, fontsize=legend_fontsize)

# Additional plot adjustments
plt.gca().set_aspect("equal", adjustable="box")
plt.grid(True)

# ===================
# Part 4: Saving Output
# ===================
# Reduce whitespace around the plot
plt.tight_layout()
plt.savefig("contour_45.pdf", bbox_inches="tight")
