
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Data configuration
np.random.seed(42)

# Grid
x = np.linspace(-200, 200, 300)
y = np.linspace(-200, 200, 300)
X, Y = np.meshgrid(x, y)

# Sensor locations (representing oceanic temperature sensors)
sensors = [(150, -150), (-150, 150), (-100, -100), (100, 100), (0, -200)]

# Ocean Temperature function
def ocean_temperature(x, y, sensors):
    Z = np.zeros_like(x)
    for sx, sy in sensors:
        Z += np.exp(-((x - sx) ** 2 + (y - sy) ** 2) / 2000) * np.random.uniform(10, 30)
    return Z

# Calculate Z as ocean temperature measure
Z = ocean_temperature(X, Y, sensors)

# Text variables
title_text = "Ocean Temperature Distribution"
xlabel_text = "Longitude (km)"
ylabel_text = "Latitude (km)"
colorbar_label_text = "Temperature (°C)"
sensor_label_format = "Sensor ({},{})"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
legend_location = "upper right"
contour_label_format = "%.1f"
fig, ax = plt.subplots(figsize=(10, 10))

# Contour
cnt = ax.contour(X, Y, Z, cmap="viridis", linewidths=1)
contour_filled = ax.contourf(X, Y, Z, cmap="viridis", alpha=0.7)
ax.clabel(cnt, cnt.levels, inline=True, fontsize=10, fmt=contour_label_format)

# Sensor locations
for sx, sy in sensors:
    ax.plot(sx, sy, "ro", markersize=10, label=sensor_label_format.format(sx, sy))

# Adding color bar
cbar = fig.colorbar(contour_filled, ax=ax)
cbar.set_label(colorbar_label_text)

# Titles and labels
ax.set_title(title_text)
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.legend(loc=legend_location)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("contour_69.pdf", bbox_inches="tight")
