
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Generate data for the plot - representing energy consumption over time
x = np.linspace(0, 24, 1000)
temperature_conditions = 5
y = [
    np.random.uniform(100, 200)
    * np.sin(0.3 * (x - i)) ** 2
    * np.exp(-0.05 * (x - i) ** 2 / np.linspace(1, 3, temperature_conditions)[index])
    for index, i in enumerate(np.linspace(4, 20, temperature_conditions))
]

# Text labels
xlabel = 'Time (hours)'
ylabel = 'Energy Consumption (kWh)'
title = 'Energy Consumption Over Time Under Various Temperature Conditions'
cbar_label = 'Temperature Condition'

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(12, 6))

# Define a new colormap for the visualization
cmap = plt.cm.viridis
sm = plt.cm.ScalarMappable(cmap=cmap, norm=plt.Normalize(vmin=0, vmax=temperature_conditions-1))
cbar = plt.colorbar(sm, ax=ax, label=cbar_label)
cbar.set_label(cbar_label, rotation=270, labelpad=25)

# Plot each condition
for i in range(temperature_conditions):
    plt.fill_between(x, y[i], color=cmap(i / temperature_conditions), alpha=0.7, label=f'Temperature Condition {i+1}')

# Add grid lines for better readability
plt.grid(True, which='both', linestyle='--', linewidth=0.5)

# Add data labels
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)
plt.legend()

# Customize axis appearance
plt.ylim(0, 250)
plt.gca().spines["top"].set_visible(False)
plt.gca().spines["right"].set_visible(False)
plt.gca().spines["bottom"].set_visible(True)
plt.gca().spines["left"].set_visible(False)
plt.gca().set_yticks([])

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_21.pdf", bbox_inches="tight")
