
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(0)

# Generate data for historical population trends (population over years)
years = np.linspace(1900, 2020, 100)  # Time in years
population_A = 1e6 * (1 + 0.01 * (years - 1950)) + 1e5 * np.random.rand(100)  # Population for City A
population_B = 1.2e6 * (1 + 0.012 * (years - 1950)) + 1e5 * np.random.rand(100)  # Population for City B

# Combine data for density estimation
data_population_A = population_A
data_population_B = population_B

# Kernel Density Estimation (KDE) for the population data
kde_population_A = gaussian_kde(data_population_A, bw_method='scott')
kde_population_B = gaussian_kde(data_population_B, bw_method='scott')

x_dens_population = np.linspace(0, max(data_population_A.max(), data_population_B.max()), 1000)
density_population_A = kde_population_A(x_dens_population)
density_population_B = kde_population_B(x_dens_population)

# Extracted variables for the density plot
xlabel_text = "Population (in Millions)"
ylabel_text = "Density"
title_text = "Density of Historical Population Trends"
label = ["City A Population Density", "City B Population Density"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Color variables
color_A = '#3498db'
color_B = '#e67e22'

# Plot the density curves for population data
ax.plot(x_dens_population, density_population_A, color=color_A, lw=2, label=label[0])
ax.fill_between(x_dens_population, density_population_A, color=color_A, alpha=0.3)

ax.plot(x_dens_population, density_population_B, color=color_B, lw=2, label=label[1])
ax.fill_between(x_dens_population, density_population_B, color=color_B, alpha=0.3)

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right')

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_26.pdf", bbox_inches="tight")
