
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
x = np.linspace(0, 14, 800)  # Time in weeks
# Simulated voter turnout data for seven different regions
turnout_data = [
    np.abs(np.random.uniform(20, 80) * (0.5 + 0.5 * np.sin(0.3 * (x - i))))
    for index, i in enumerate(np.linspace(0, 6, 7))
]

# Combine data for density estimation
combined_turnout = np.concatenate(turnout_data)

# Kernel Density Estimation (KDE) for the turnout data
kde_turnout = gaussian_kde(combined_turnout, bw_method='scott')
x_dens_turnout = np.linspace(0, max(combined_turnout), 1000)
density_turnout = kde_turnout(x_dens_turnout)

# Create density plots for individual region data
density_curves = []
for turnout in turnout_data:
    kde_region = gaussian_kde(turnout, bw_method='scott')
    density_curves.append(kde_region(x_dens_turnout))

# Extracted variables for the density plot
xlabel_text = "Voter Turnout (%)"
ylabel_text = "Density"
title_text = "Density of Voter Turnout (%) Across Different Regions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 5))

# Plot the density curve for overall turnout data
ax.plot(x_dens_turnout, density_turnout, color='purple', lw=2, label='Overall Turnout Density')
ax.fill_between(x_dens_turnout, density_turnout, color='purple', alpha=0.3)

# Plot density area for an additional region
additional_region_data = np.abs(np.random.uniform(30, 70) * (0.5 + 0.5 * np.sin(0.4 * (x - 4))))
kde_additional_region = gaussian_kde(additional_region_data, bw_method='scott')
density_additional_region = kde_additional_region(x_dens_turnout)
ax.fill_between(x_dens_turnout, density_additional_region, color='orange', alpha=0.5, label='Region 8 Turnout Density')

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right', fontsize=10)

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_36.pdf", bbox_inches="tight")
