
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)  # Set a seed for reproducibility
from scipy.stats import gaussian_kde

# Generate data for the plot
x = np.linspace(1, 14, 14)  # Days
# Simulated daily viewership growth data for seven different channels
y = [
    np.random.uniform(100, 500) + np.random.normal(scale=30, size=x.shape).cumsum()
    for _ in range(7)
]

# Combine all channel data into a single array for density estimation
all_data = np.concatenate(y)

# Extracted text variables
xlabel_text = "Viewership Growth"
ylabel_text = "Density"
title_text = "Density of Daily Viewership Growth Across Channels Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Perform Kernel Density Estimation (KDE) using scipy
kde = gaussian_kde(all_data, bw_method='scott')
x_dens = np.linspace(min(all_data), max(all_data), 1000)
density = kde(x_dens)

# Plot the density curve
ax.plot(x_dens, density, color='blue', lw=2)
ax.fill_between(x_dens, density, color='blue', alpha=0.3)  # Fill under the curve

# Set labels and title
ax.set_xlabel(xlabel_text)
ax.set_ylabel(ylabel_text)
ax.set_title(title_text)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.spines["bottom"].set_visible(True)
ax.spines["left"].set_visible(True)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)

# Add grid for readability
plt.grid(True, linestyle=":", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_37.pdf", bbox_inches="tight")
