# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(24)  # Different seed for new data
from scipy.stats import gaussian_kde

# Generate data for the plot
x = np.linspace(0, 15, 1000)  # Time in days

# Simulated wind speed data for seven different locations with different parameters
y_data = [
    np.abs(
        np.random.uniform(5, 20)
        * np.sin(0.2 * x * (index + 1))
        * np.exp(-0.03 * (x - i) ** 2 / np.linspace(0.2, 1.5, 7)[index])
        + np.random.normal(0, 1.5, x.shape)
    )  # Ensure positive values and add noise for complexity
    for index, i in enumerate(np.linspace(1, 12, 7))
]

# Select two sets of location data for density plotting
y_data_1 = y_data[:4]  # First 4 locations
y_data_2 = y_data[4:]  # Last 3 locations

# Extracted variables
xlabel_text = "Wind Speed (m/s)"
ylabel_text = "Density"
title_text_1 = "Density of Wind Speed for Locations (Set 1)"
title_text_2 = "Density of Wind Speed for Locations (Set 2)"
legend_title = "Locations"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, axes = plt.subplots(2, 1, figsize=(12, 10))

# New color schemes for density curves
colors_1 = ["#ff9999", "#66b3ff", "#99ff99", "#ffcc99"]  # Colors for first set
colors_2 = ["#c2c2f0", "#ffb3e6", "#c4e17f"]  # Colors for second set

# Plot separate density curves for the first set of locations
for i, y in enumerate(y_data_1):
    kde = gaussian_kde(y, bw_method="scott")
    x_dens = np.linspace(min(y), max(y), 1000)
    density = kde(x_dens)
    axes[0].plot(x_dens, density, color=colors_1[i], lw=2, label=f"Location {i+1}")
    axes[0].fill_between(x_dens, density, color=colors_1[i], alpha=0.2)

# Plot separate density curves for the second set of locations
for i, y in enumerate(y_data_2, start=4):
    kde = gaussian_kde(y, bw_method="scott")
    x_dens = np.linspace(min(y), max(y), 1000)
    density = kde(x_dens)
    axes[1].plot(x_dens, density, color=colors_2[i - 4], lw=2, label=f"Location {i+1}")
    axes[1].fill_between(x_dens, density, color=colors_2[i - 4], alpha=0.2)

# Set labels and title for the first subplot
axes[0].set_xlabel(xlabel_text)
axes[0].set_ylabel(ylabel_text)
axes[0].set_title(title_text_1)
axes[0].legend(title=legend_title, loc="upper right", fontsize=10)
axes[0].grid(True, linestyle=":", alpha=0.5)
axes[0].spines["top"].set_visible(False)
axes[0].spines["right"].set_visible(False)

# Set labels and title for the second subplot
axes[1].set_xlabel(xlabel_text)
axes[1].set_ylabel(ylabel_text)
axes[1].set_title(title_text_2)
axes[1].legend(title=legend_title, loc="upper right", fontsize=10)
axes[1].grid(True, linestyle=":", alpha=0.5)
axes[1].spines["top"].set_visible(False)
axes[1].spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_41.pdf", bbox_inches="tight")
