
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
cities = ['New York', 'Los Angeles', 'Chicago', 'Houston', 'Phoenix']
# Simulated traffic volume data for each city over a year (vehicles per day)
traffic_volumes = [
    np.random.normal(20000 + i*5000, 4000, 365)  # Mean traffic volume varies by city
    for i in range(len(cities))
]

# Adding another set of traffic volume data for variation
more_traffic_volumes = [
    np.random.normal(25000 + i*3000, 3000, 365)  # Another set with different mean and variance
    for i in range(3)  # Only add a few extra cities to avoid too many lines
]

# Combine data for density estimation
traffic_data = np.concatenate(traffic_volumes + more_traffic_volumes)

# Kernel Density Estimation (KDE) for the traffic volume data
kde_traffic = gaussian_kde(traffic_data, bw_method='scott')
x_dens_traffic = np.linspace(min(traffic_data), max(traffic_data), 1000)
density_traffic = kde_traffic(x_dens_traffic)

# Create density plots for individual city data
density_curves = []
for traffic in traffic_volumes + more_traffic_volumes:
    kde_city = gaussian_kde(traffic, bw_method='scott')
    density_curves.append(kde_city(x_dens_traffic))

# Extracted variables for the density plot
xlabel_text = "Traffic Volume (Vehicles per Day)"
ylabel_text = "Density"
title_text = "Density of Daily Traffic Volume Across Different Cities"
label = "Overall Traffic Volume Density"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Plot the density curve for overall traffic volume data
ax.plot(x_dens_traffic, density_traffic, color='blue', lw=2, label=label)
ax.fill_between(x_dens_traffic, density_traffic, color='blue', alpha=0.3)

# Plot individual city density curves (fewer lines)
colors = ['#1f77b4', '#ff7f0e', '#2ca02c', '#d62728', '#9467bd']
for i, density in enumerate(density_curves):
    ax.plot(x_dens_traffic, density, color=colors[i % len(colors)], lw=1.5, alpha=0.7, label=f'City {i+1} Density')

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right', fontsize=10)

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_43.pdf", bbox_inches="tight")
