
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
appliances = ['Heater', 'Air Conditioner', 'Refrigerator', 'Lights', 'Washing Machine', 'Oven', 'Television']
# Simulated energy consumption data for each appliance (in kilowatt-hours)
consumptions = [
    np.random.normal(300 + i*50, 80, 300)  # Mean consumption varies by appliance
    for i in range(len(appliances))
]

# Adding one more set of consumption data for variation
additional_consumption = np.random.normal(600, 100, 300)  # A different set with a higher mean and variance

# Combine data for density estimation
consumption_data = np.concatenate(consumptions + [additional_consumption])

# Kernel Density Estimation (KDE) for the energy consumption data
kde_consumption = gaussian_kde(consumption_data, bw_method='scott')
x_dens_consumption = np.linspace(min(consumption_data), max(consumption_data), 1000)
density_consumption = kde_consumption(x_dens_consumption)

# Create density plot for the additional consumption data
kde_additional = gaussian_kde(additional_consumption, bw_method='scott')
density_additional = kde_additional(x_dens_consumption)

# Extracted variables for the density plot
xlabel_text = "Energy Consumption (Kilowatt-hours)"
ylabel_text = "Density"
title_text = "Density of Energy Consumption with Additional Consumption Distribution"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Plot the density curve for overall consumption data
ax.plot(x_dens_consumption, density_consumption, color='green', lw=2, label='Overall Consumption Density')
ax.fill_between(x_dens_consumption, density_consumption, color='green', alpha=0.3)

# Plot the density area for additional consumption data
ax.fill_between(x_dens_consumption, density_additional, color='orange', alpha=0.5, label='Additional Consumption Density')

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right', fontsize=10)

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_44.pdf", bbox_inches="tight")
