
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
regions = ['Region A', 'Region B']
# Simulated crop yield (tons per hectare)
crop_yield_A = np.random.normal(7, 1.5, 300)
crop_yield_B = np.random.normal(5, 1.0, 300)

# Combine data for density estimation
yield_data_1 = crop_yield_A
yield_data_2 = crop_yield_B

# Kernel Density Estimation (KDE) for the crop yield data
kde_yield_1 = gaussian_kde(yield_data_1, bw_method='scott')
kde_yield_2 = gaussian_kde(yield_data_2, bw_method='scott')

x_dens_yield = np.linspace(min(min(yield_data_1), min(yield_data_2)), 
                           max(max(yield_data_1), max(yield_data_2)), 1000)
density_yield_1 = kde_yield_1(x_dens_yield)
density_yield_2 = kde_yield_2(x_dens_yield)

# Extracted variables for the density plot
xlabel_text = "Crop Yield (tons/hectare)"
ylabel_text = "Density"
title_text = "Density of Crop Yield in Different Regions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density curves for crop yield data
ax.plot(x_dens_yield, density_yield_1, color='purple', lw=2, label='Region A Yield Density')
ax.fill_between(x_dens_yield, density_yield_1, color='purple', alpha=0.3)

ax.plot(x_dens_yield, density_yield_2, color='orange', lw=2, label='Region B Yield Density')
ax.fill_between(x_dens_yield, density_yield_2, color='orange', alpha=0.3)

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc='upper right')

# ===================
# Part 4: Saving Output for Density Plot
# ===================
plt.tight_layout()
plt.savefig("density_46.pdf", bbox_inches="tight")
