# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(42)
metrics = ["Global Temperature Anomaly", "CO2 Concentration", "Arctic Sea Ice Extent"]
data = [np.random.normal(loc=0.5 + i * 0.3, scale=0.1, size=300) for i in range(3)]

# Combine all environmental metrics data for density estimation
all_data = np.concatenate(data)

# Kernel Density Estimation (KDE) for the data
kde_data = gaussian_kde(all_data, bw_method="scott")
x_dens_data = np.linspace(min(all_data), max(all_data), 1000)
density_data = kde_data(x_dens_data)

# Extracted variables for the density plot
xlabel_text = "Environmental Metric Value"
ylabel_text = "Density"
title_text = "Density of Environmental Metrics Over Time"

label = "Metrics Density"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 5))

# Plot the density curve for environmental metrics data
ax.plot(x_dens_data, density_data, color="purple", lw=2, label=label)
ax.fill_between(x_dens_data, density_data, color="purple", alpha=0.3)

# Set labels and title
ax.set_xlabel(xlabel_text, fontsize=12)
ax.set_ylabel(ylabel_text, fontsize=12)
ax.set_title(title_text, fontsize=16)

# Customize plot border and ticks
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
ax.tick_params(left=True, labelleft=True, bottom=True, labelbottom=True)
ax.grid(True, linestyle=":", alpha=0.5)

# Adding a legend
ax.legend(loc="upper right")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_56.pdf", bbox_inches="tight")
