
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Fix a seed for reproducibility
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(62)

# Generate bimodal distribution for religious service attendance
# Simulate attendance frequency for two groups (Devout and Casual)
attendance_devout = np.random.normal(loc=3, scale=1, size=400)  # Devout attendees
attendance_casual = np.random.normal(loc=1, scale=0.5, size=400)  # Casual attendees
attendance_data = np.concatenate([attendance_devout, attendance_casual])

# X-axis values for KDE
xs = np.linspace(0, 7, 200)

# Axes Limits and Labels
title = "KDE Plot of Religious Service Attendance Distribution"
xlabel_value = "Attendance Frequency (per week)"
ylabel_value = "Density"
legend_label = "Attendance Density for Devout and Casual Attendees"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
xticks_values = [0, 1, 2, 3, 4, 5, 6, 7]
xticklabels = ["0", "1", "2", "3", "4", "5", "6", "7"]
yticks_values = [0, 0.1, 0.2, 0.3, 0.4]
yticklabels = ["0", "0.1", "0.2", "0.3", "0.4"]
xlim_values = [0, 7]
ylim_values = [0, 0.45]
title_fontsize = 14
title_weight = "bold"
label_fontsize = 12
legend_loc = "upper right"
legend_fontsize = 12

# Set the figure size
fig, ax = plt.subplots(figsize=(10, 6))

# Create the KDE plot with adjusted x-axis range
density = gaussian_kde(attendance_data)
density.covariance_factor = lambda: 0.3
density._compute_covariance()
plt.fill_between(xs, density(xs), color="#66c2a5", edgecolor="#fc8d62")

# Customize ticks and labels
ax.set_xticks(xticks_values)
ax.set_xticklabels(xticklabels, fontsize=label_fontsize)

ax.set_yticks(yticks_values)
ax.set_yticklabels(yticklabels, fontsize=label_fontsize)

plt.xlim(xlim_values)
plt.ylim(ylim_values)

# Set the title and labels
plt.title(title, fontsize=title_fontsize, weight=title_weight)
plt.xlabel(xlabel_value, fontsize=label_fontsize)
plt.ylabel(ylabel_value, fontsize=label_fontsize)

# Add a legend
ax.legend([legend_label], loc=legend_loc, fontsize=legend_fontsize)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_62.pdf", bbox_inches="tight")
