
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(67)

# Generate sample temperature data for three cities
city_a_temps = np.random.normal(loc=75, scale=10, size=365)  # in Fahrenheit
city_b_temps = np.random.normal(loc=65, scale=15, size=365)  # in Fahrenheit
city_c_temps = np.random.normal(loc=85, scale=7, size=365)   # in Fahrenheit

labels = ["City A Temps", "City B Temps", "City C Temps"]
avxlabel = "Average City A Temperature"
xlabel = "Temperature (°F)"
ylabel = "Density"
title = "Temperature Density Distributions Across Three Cities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_linestyle = "-"
grid_linewidth = 0.5
grid_alpha = 0.7
line_linestyle = "-."
line_linewidth = 2
fill_alpha = 0.25

fig, ax = plt.subplots(figsize=(10, 6))

# Plot the density plots
for data, color, label in zip(
    [city_a_temps, city_b_temps, city_c_temps],
    ["magenta", "cyan", "limegreen"],
    labels,
):
    density = gaussian_kde(data)
    xs = np.linspace(min(data) - 10, max(data) + 10, 200)
    density.covariance_factor = lambda: 0.25  # Adjust for smoother curves
    density._compute_covariance()
    plt.plot(
        xs,
        density(xs),
        color=color,
        label=label,
        linestyle=line_linestyle,  # dash-dot line
        linewidth=line_linewidth,  # slightly thicker lines
    )
    plt.fill_between(
        xs, density(xs), color=color, alpha=fill_alpha
    )  # fill for shade

# Plot a vertical line indicating the average temperature of City A
plt.axvline(
    x=np.mean(city_a_temps),
    color="red",
    linestyle=line_linestyle,
    linewidth=line_linewidth,
    label=avxlabel,
)

# Set labels and title
ax.set_xlim(
    min(min(city_a_temps), min(city_b_temps), min(city_c_temps)) - 10,
    max(max(city_a_temps), max(city_b_temps), max(city_c_temps)) + 10,
)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Show grid
plt.grid(True, linestyle=grid_linestyle, linewidth=grid_linewidth, alpha=grid_alpha)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("density_67.pdf", bbox_inches="tight")
