
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
from scipy.stats import gaussian_kde

# Reproducibility
np.random.seed(78)

# Simulated stock price data (in $)
company1 = np.random.normal(loc=150, scale=20, size=1000)
company2 = np.random.normal(loc=200, scale=25, size=1000)
company3 = np.random.normal(loc=180, scale=15, size=1000)

# Compute density for each dataset
density1 = gaussian_kde(company1)
density2 = gaussian_kde(company2)
density3 = gaussian_kde(company3)
xs = np.linspace(75, 275, 400)
ys1 = density1(xs)
ys2 = density2(xs)
ys3 = density3(xs)

# Labels
labels = ["Company A", "Company B", "Company C"]
xlabel = "Stock Price ($)"
ylabel = "Density"
title = "Density Plot of Stock Prices Across Companies"
peak_annotation = "Peak Price: ${} "

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_style = "--"
# Create the figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

# Fill between x for density regions with optimized colors
plt.fill_between(xs, ys1, color="lightgreen", alpha=0.5, label=labels[0])
plt.fill_between(xs, ys2, color="lightblue", alpha=0.5, label=labels[1])
plt.fill_between(xs, ys3, color="palegoldenrod", alpha=0.5, label=labels[2])

# Plot lines for densities with optimized styles
plt.plot(xs, ys1, color="darkgreen", linestyle="-.", linewidth=2)
plt.plot(xs, ys2, color="darkblue", linestyle="-.", linewidth=2)
plt.plot(xs, ys3, color="goldenrod", linestyle="-.", linewidth=2)

# Set labels and title (if any)
ax.set_xlim(75, 275)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Annotations for peaks
peak1 = xs[np.argmax(ys1)]
peak2 = xs[np.argmax(ys2)]
peak3 = xs[np.argmax(ys3)]
plt.annotate(
    peak_annotation.format(int(peak1)),
    xy=(peak1, max(ys1)),
    xytext=(peak1 + 10, max(ys1) - 0.002),
    arrowprops=dict(facecolor="black", shrink=0.05),
)
plt.annotate(
    peak_annotation.format(int(peak2)),
    xy=(peak2, max(ys2)),
    xytext=(peak2 + 10, max(ys2) + 0.002),
    arrowprops=dict(facecolor="black", shrink=0.05),
)
plt.annotate(
    peak_annotation.format(int(peak3)),
    xy=(peak3, max(ys3)),
    xytext=(peak3 + 10, max(ys3) - 0.002),
    arrowprops=dict(facecolor="black", shrink=0.05),
)

# Show grid with custom style
plt.grid(True, linestyle=grid_style)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_78.pdf", bbox_inches="tight")
