
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Reproducibility
import numpy as np
from scipy.stats import gaussian_kde

np.random.seed(79)  # New seed for new data generation
# Simulated public approval rating data
approval_leader1 = np.random.normal(loc=45, scale=10, size=1000)
approval_leader2 = np.random.normal(loc=55, scale=15, size=1000)
approval_leader3 = np.random.normal(loc=65, scale=20, size=1000)

# Compute density for each dataset
density1 = gaussian_kde(approval_leader1)
density2 = gaussian_kde(approval_leader2)
density3 = gaussian_kde(approval_leader3)
xs = np.linspace(0, 100, 400)
ys1 = density1(xs)
ys2 = density2(xs)
ys3 = density3(xs)

# Labels
labels = ["Leader 1", "Leader 2", "Leader 3"]
xlabel = "Approval Rating"
ylabel = "Density"
title = "Density Plot of Public Approval Ratings for Political Leaders"
peak_annotation = "Peak: {}"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
grid_style = "--"
# Create the figure and axis
fig, ax = plt.subplots(figsize=(10, 6))

# Fill between x for density regions with optimization
plt.fill_between(xs, ys1, color="skyblue", alpha=0.5, label=labels[0])
plt.fill_between(xs, ys2, color="lightgreen", alpha=0.5, label=labels[1])
plt.fill_between(xs, ys3, color="lightcoral", alpha=0.5, label=labels[2])

# Plot lines for densities
plt.plot(xs, ys1, color="blue", linestyle="-.", linewidth=2)
plt.plot(xs, ys2, color="green", linestyle="-.", linewidth=2)
plt.plot(xs, ys3, color="red", linestyle="-.", linewidth=2)

# Set labels and title (if any)
ax.set_xlim(0, 100)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Annotations for peaks
peak1 = xs[np.argmax(ys1)]
peak2 = xs[np.argmax(ys2)]
peak3 = xs[np.argmax(ys3)]
plt.annotate(
    peak_annotation.format(int(peak1)),
    xy=(peak1, max(ys1)),
    xytext=(peak1 + 5, max(ys1) - 0.005),
    arrowprops=dict(facecolor="black", shrink=0.05),
)
plt.annotate(
    peak_annotation.format(int(peak2)),
    xy=(peak2, max(ys2)),
    xytext=(peak2 + 5, max(ys2) + 0.005),
    arrowprops=dict(facecolor="black", shrink=0.05),
)
plt.annotate(
    peak_annotation.format(int(peak3)),
    xy=(peak3, max(ys3)),
    xytext=(peak3 + 5, max(ys3) - 0.005),
    arrowprops=dict(facecolor="black", shrink=0.05),
)

# Show grid
plt.grid(True, linestyle=grid_style)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("density_79.pdf", bbox_inches="tight")
