
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.colors as mcolors

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for factors affecting transportation efficiency
categories = [
    "Infrastructure Quality",
    "Traffic Conditions",
    "Public Transport Availability",
    "Fuel Costs",
    "Weather Conditions",
    "Regulatory Policies",
    "Technological Innovations",
    "Driver Experience",
    "Vehicle Maintenance",
    "Urban Planning",
]
values = [0.30, -0.25, 0.15, 0.05, -0.10, 0.20, 0.35, -0.05, 0.10, -0.15]
errors = [0.05, 0.04, 0.03, 0.02, 0.04, 0.03, 0.05, 0.02, 0.04, 0.03]

min_val = min(values) - 0.1
max_val = max(values) + 0.1

# Axes Limits and Labels
ylabel_value = "Factors in Transportation"
xlabel_value = "Impact on Transport Efficiency (Δ to control)"

title = "Factors Affecting Transportation Efficiency"
suptitle = "A Study on Various Influences in the Transport Sector"
legendlabels = "Impact of Factors"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(figsize=(12, 8))

# Normalizing function to convert values to a 0-1 range for color scaling
def normalize(value, min_val, max_val):
    return (value - min_val) / (max_val - min_val)

# Determine color based on normalized value (using shades of blue and grey)
def get_color(value):
    norm_value = normalize(value, min_val, max_val)
    blue_base = np.array(mcolors.to_rgb("#1f77b4"))
    grey_base = np.array(mcolors.to_rgb("#7f7f7f"))
    # Create a color that ranges from light grey to blue
    return mcolors.to_hex(grey_base * (1 - norm_value) + blue_base * norm_value)

colors = [get_color(value) for value in values]

# Horizontal bar chart
bars = ax.barh(
    categories, values, xerr=errors, color=colors, capsize=5, edgecolor="darkgrey", alpha=0.85
)
ax.set_ylabel(ylabel_value, fontsize=14)
ax.set_xlabel(xlabel_value, fontsize=14)
ax.set_title(title, fontsize=16)
fig.suptitle(suptitle, fontsize=18, fontweight='bold')

# Set y-axis limits and x-axis limits
ax.set_xlim(min_val, max_val)  # Adjust limits to encompass errors

# Remove top and right spines for a cleaner look
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)

# Customize grid lines
ax.xaxis.grid(True, linestyle="--", which="major", color="gray", alpha=0.6)
ax.set_axisbelow(True)

# Annotating bars with values
for bar, value, error in zip(bars, values, errors):
    ax.text(bar.get_width() + error + 0.02, bar.get_y() + bar.get_height() / 2,
            f'{value:.2f}', va='center', ha='left', fontsize=12, color='black')

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout to prevent clipping of ylabel
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("errorbar_113.pdf", bbox_inches="tight")
