# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Environmental Metrics Data for three major cities
metrics = ["Pollution Levels", "Green Space Coverage", "Renewable Energy Usage"]
values = np.array(
    [
        [70, 30, 60],  # New York
        [80, 40, 50],  # Los Angeles
        [60, 35, 55],  # Chicago
    ]
)

# Asymmetric error values, proportionate to the data scale
errors = np.array(
    [
        [[7, 6], [5, 4], [6, 5]],  # Errors for New York (lower, upper)
        [[8, 7], [6, 5], [5, 4]],  # Errors for Los Angeles
        [[6, 7], [4, 5], [5, 6]],  # Errors for Chicago
    ]
)

# Creating subplots for each city
cities = ["New York", "Los Angeles", "Chicago"]

ylabel = "Environmental Metric Values"
ylim = [20, 90]
suptitle = "Environmental Metrics in Major Cities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, axs = plt.subplots(1, 3, figsize=(12, 5))  # Slightly larger layout


# Function to plot each city's data
def plot_city_data(ax, errors, city_index, city_name):
    x = np.arange(len(metrics))  # the label locations
    bar_colors = ["#ff9999", "#66b3ff", "#99ff99"]
    barerrors = np.array(errors).T[:, :, city_index]
    bars = ax.bar(x, values[city_index], yerr=barerrors, color=bar_colors, capsize=5)
    for bar, lower_error, upper_error in zip(bars, barerrors[0], barerrors[1]):
        # Position for lower error text
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            bar.get_height() - lower_error - 2,
            f"-{lower_error}",
            va="bottom",
            ha="center",
            color="black",
            fontsize=8,
        )
        # Position for upper error text
        ax.text(
            bar.get_x() + bar.get_width() / 2,
            bar.get_height() + upper_error + 1,
            f"+{upper_error}",
            ha="center",
            color="black",
            fontsize=8,
        )

    ax.set_title(city_name)
    ax.set_xticks(x)
    ax.set_xticklabels(metrics, rotation=45, ha="right")
    ax.set_ylabel(ylabel)
    ax.set_ylim(ylim)  # Uniform scale for all charts


# Plotting the data for each city
for i, city in enumerate(cities):
    plot_city_data(axs[i], errors, i, city)

# Adding a super title
plt.suptitle(suptitle, fontsize=16)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(
    rect=[0, 0.03, 1, 0.95]
)  # Adjust tight_layout to make space for supertitle
plt.savefig("errorbar_115.pdf", bbox_inches="tight")
