# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Data for plotting
players = [
    "Player A",
    "Player B",
    "Player C",
    "Player D",
    "Player E",
    "Player F",
    "Player G",
    "Player H",
    "Player I",
    "Player J",
]
mean_scores = [22.3, 18.1, 25.5, 19.8, 24.2, 21.0, 26.1, 20.3, 23.2, 27.5]
errors = [
    [2.2, -2.5],
    [1.8, -2.1],
    [2.0, -1.9],
    [1.5, -1.8],
    [2.3, -2.2],
    [1.9, -1.7],
    [2.1, -2.0],
    [1.6, -1.4],
    [2.2, -1.9],
    [1.7, -1.5],
]
annotations = [
    r"$%.1f^{+%.1f} _{-%.1f}$" % (mean_scores[i], errors[i][0], abs(errors[i][1]))
    for i in range(len(mean_scores))
]
xticks = np.arange(15, 30, 2)
xlim = [15, 30]
xvline = 20
xvspan = [18, 22]

# Chart labels
xlabel = "Average Points Per Game"
title = "Basketball Player Performance Analysis"
labels = ["Target Range", "Threshold Value"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 8))

# Error bars with different positive and negative values
for i, (player, mean_score, error, annotation) in enumerate(
    zip(players, mean_scores, errors, annotations)
):
    ax.errorbar(
        mean_score,
        i,
        xerr=[[abs(error[1])], [error[0]]],
        fmt="o",
        color="#1f77b4",  # Custom color scheme
        ecolor="#ff7f0e",  # Custom error color
        capsize=5,
    )
    ax.text(
        mean_score,
        i - 0.15,
        annotation,
        va="center",
        ha="center",
        fontsize=9,
    )

# Highlighted region with custom color and alpha
ax.axvspan(xvspan[0], xvspan[1], color="lightgreen", alpha=0.3, label=labels[0])

# Set labels and title
ax.set_yticks(range(len(players)))
ax.set_yticklabels(players)
ax.set_xlabel(xlabel, fontsize=12)
ax.set_xlim(xlim)
ax.invert_yaxis()  # Invert y-axis to match the original image
ax.axvline(x=xvline, linestyle="--", color="red", label=labels[1])

# Adjust x-axis ticks and labels
ax.set_xticks(xticks)
ax.set_xticklabels([f"{x:.1f}" for x in xticks])
ax.set_title(title, fontsize=14)
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("errorpoint_28.pdf", bbox_inches="tight")
