

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(42)

# Categories of watch times and their labels
media_types = ["TV Shows", "Movies", "Documentaries"]

# Monthly x positions
x_positions = np.linspace(-20, 20, 6)
tv_shows_watch_time = np.random.uniform(100, 200, 6)
movies_watch_time = np.random.uniform(80, 150, 6)
documentaries_watch_time = np.random.uniform(50, 100, 6)

# Random error margins (positive and negative)
tv_shows_error = [np.random.uniform(5, 10, 6), np.random.uniform(5, 10, 6)]
movies_error = [np.random.uniform(3, 7, 6), np.random.uniform(3, 7, 6)]
documentaries_error = np.random.uniform(2, 5, 6)

# Label positions
vertical_line_position = 2.5

# Titles for each subplot
titles = ["Average TV Shows Watch Time", "Average Movies Watch Time", "Average Documentaries Watch Time"]

# Axis labels
xlabel = "Months"
ylabel = "Watch Time in Hours"

suptitle = "Average Monthly Watch Time Over Six Months"
filename = "errorpoint_76_1.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with three subplots and a shared x-axis
fig, (ax0, ax1, ax2) = plt.subplots(figsize=(8, 10), nrows=3, sharex=True)

# Color maps and marker styles
tv_shows_colors = plt.get_cmap("Pastel1")(np.linspace(0, 1, 6))
movies_colors = plt.get_cmap("Pastel2")(np.linspace(0, 1, 6))
documentaries_color = "#8dd3c7"

# First subplot with symmetric vertical error bars
for i in range(len(x_positions)):
    ax0.errorbar(
        x_positions[i],
        tv_shows_watch_time[i],
        yerr=[[tv_shows_error[0][i]], [tv_shows_error[1][i]]],
        fmt="o",
        color=tv_shows_colors[i],
        capsize=4,
    )
    ax0.text(
        x_positions[i] - 0.1,
        tv_shows_watch_time[i] + 2,
        f"{tv_shows_watch_time[i]:.2f}",
        fontsize=8,
        ha="right",
    )
ax0.set_title(titles[0])
ax0.axhline(y=150, linestyle="--", color="#FF6347")
ax0.yaxis.grid(True)
ax0.xaxis.grid(False)

# Second subplot with symmetric horizontal error bars
for i in range(len(x_positions)):
    ax1.errorbar(
        x_positions[i],
        movies_watch_time[i],
        xerr=[[movies_error[0][i]], [movies_error[1][i]]],
        fmt="s",
        color=movies_colors[i],
        capsize=4,
    )
    ax1.text(
        x_positions[i] + 0.1,
        movies_watch_time[i] + 1,
        f"{movies_watch_time[i]:.2f}",
        fontsize=8,
        ha="left",
    )
ax1.set_title(titles[1])
ax1.axvline(x=vertical_line_position, linestyle="--", color="#4682B4")
ax1.xaxis.grid(True)
ax1.yaxis.grid(False)

# Third subplot with symmetric vertical error bars
ax2.errorbar(
    x_positions,
    documentaries_watch_time,
    yerr=documentaries_error,
    fmt="*",
    color=documentaries_color,
    capsize=4,
)
ax2.set_title(titles[2])
ax2.legend([media_types[2]], loc="upper left")
ax2.yaxis.grid(True)
ax2.xaxis.grid(False)

# Set common labels
fig.supxlabel(xlabel)
fig.supylabel(ylabel)
fig.suptitle(suptitle, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.savefig("errorpoint_76.pdf", bbox_inches="tight")

