
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Set random seed
np.random.seed(0)

# Define the types of air quality measurements
air_quality_metrics = ["PM2.5", "PM10", "NO2"]

# Define the x positions for the data points
x_positions = np.linspace(0, 20, 6)

# Generate random air quality measurement data for each metric
pm25_levels = np.random.uniform(20, 100, 6)
pm10_levels = np.random.uniform(30, 120, 6)
no2_levels = np.random.uniform(10, 70, 6)

# Generate random error values for each metric
pm25_error = [np.random.uniform(2, 6, 6), np.random.uniform(2, 6, 6)]
pm10_error = [np.random.uniform(3, 8, 6), np.random.uniform(3, 8, 6)]
no2_error = np.random.uniform(1, 4, 6)

# Define plot line positions and titles
horizontal_line_position = 60
vertical_line_position = 10

# Titles for each subplot
titles = ["PM2.5 Levels", "PM10 Levels", "NO2 Levels"]

# Axis labels and other texts
xlabel = "Monitoring Points"
ylabel_pm25 = "PM2.5 (µg/m³)"
ylabel_pm10 = "PM10 (µg/m³)"
ylabel_no2 = "NO2 (ppb)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with three subplots and shared x-axis
fig, (ax0, ax1, ax2) = plt.subplots(figsize=(6, 9), nrows=3, sharex=True)

# Define color schemes for each air quality metric
pm25_colors = plt.get_cmap("YlOrBr")(np.linspace(0.2, 0.8, 6))
pm10_colors = plt.get_cmap("Blues")(np.linspace(0.2, 0.8, 6))
no2_color = "#228B22"

# First subplot: PM2.5 Levels with symmetric vertical error bars
for i in range(len(x_positions)):
    ax0.errorbar(
        x_positions[i],
        pm25_levels[i],
        yerr=[[pm25_error[0][i]], [pm25_error[1][i]]],
        fmt="o",
        color=pm25_colors[i],
        capsize=4,
    )
    ax0.text(
        x_positions[i] - 0.5,
        pm25_levels[i],
        f"{pm25_levels[i]:.2f}",
        fontsize=8,
        ha="right",
    )
ax0.set_title(titles[0])
ax0.axhline(y=horizontal_line_position, linestyle="--", color="#FFA500")
ax0.set_ylabel(ylabel_pm25)
ax0.yaxis.grid(True)
ax0.xaxis.grid(False)

# Second subplot: PM10 Levels with symmetric horizontal error bars
for i in range(len(x_positions)):
    ax1.errorbar(
        x_positions[i],
        pm10_levels[i],
        xerr=[[pm10_error[0][i]], [pm10_error[1][i]]],
        fmt="o",
        color=pm10_colors[i],
        capsize=4,
    )
    ax1.text(
        x_positions[i] + 0.5,
        pm10_levels[i] + 0.1,
        f"{pm10_levels[i]:.2f}",
        fontsize=8,
        ha="left",
    )
ax1.set_title(titles[1])
ax1.axvline(x=vertical_line_position, linestyle="--", color="#1E90FF")
ax1.set_ylabel(ylabel_pm10)
ax1.xaxis.grid(True)
ax1.yaxis.grid(False)

# Third subplot: NO2 Levels with symmetric vertical error bars
ax2.errorbar(
    x_positions,
    no2_levels,
    yerr=no2_error,
    fmt="*",
    color=no2_color,
    capsize=4,
)
ax2.set_title(titles[2])
ax2.set_ylabel(ylabel_no2)
ax2.set_xlabel(xlabel)
ax2.legend([air_quality_metrics[2]], loc="upper left")
ax2.yaxis.grid(True)
ax2.xaxis.grid(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("errorpoint_77.pdf", bbox_inches="tight")
