

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
months = [
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec",
]

x = np.arange(len(months))  # Months
gdp_growth = np.random.uniform(1, 4, len(months))
unemployment_rate = np.random.uniform(4, 8, len(months))
inflation_rate = np.random.uniform(1, 3, len(months))
error_gdp_growth = [
    np.random.uniform(0.2, 0.5, len(months)),
    np.random.uniform(0.2, 0.5, len(months)),
]
error_unemployment_rate = [
    np.random.uniform(0.5, 1.0, len(months)),
    np.random.uniform(0.5, 1.0, len(months)),
]  # Symmetric horizontal error
error_inflation_rate = np.random.uniform(0.1, 0.3, len(months))
chart_title = "Monthly Economic Indicators with Error Bars"
highlight_month = 6  # July

ylabel = [
    "GDP Growth Rate (%)",
    "Unemployment Rate (%)",
    "Inflation Rate (%)",
]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with three subplots and shared x-axis
fig, (ax0, ax1, ax2) = plt.subplots(figsize=(8, 12), nrows=3, sharex=True)
colors_gdp_growth = plt.get_cmap("Blues")(np.linspace(0.5, 1.0, len(months)))
colors_unemployment_rate = plt.get_cmap("Oranges")(np.linspace(0.5, 1.0, len(months)))
color_inflation_rate = "darkgreen"

# First subplot with symmetric vertical error bars (GDP Growth Rate)
for i in range(len(x)):
    ax0.errorbar(
        x[i],
        gdp_growth[i],
        yerr=[[error_gdp_growth[0][i]], [error_gdp_growth[1][i]]],
        fmt="o",
        color=colors_gdp_growth[i],
        capsize=4,
    )
    ax0.text(
        x[i] - 0.2,
        gdp_growth[i] + 0.1,
        f"{gdp_growth[i]:.2f}",
        fontsize=8,
        ha="right",
    )
ax0.set_title(chart_title)
ax0.axhline(y=2.5, linestyle="--", color="#6b8e23")
ax0.set_ylabel(ylabel[0])
ax0.yaxis.grid(True)
ax0.xaxis.grid(False)

# Second subplot with symmetric horizontal error bars (Unemployment Rate)
for i in range(len(x)):
    ax1.errorbar(
        x[i],
        unemployment_rate[i],
        xerr=[[error_unemployment_rate[0][i]], [error_unemployment_rate[1][i]]],
        fmt="o",
        color=colors_unemployment_rate[i],
        markersize=8,
    )
    ax1.text(
        x[i] + 0.1,
        unemployment_rate[i] + 0.2,
        f"{unemployment_rate[i]:.2f}",
        fontsize=8,
        ha="left",
    )
ax1.axvline(x=highlight_month, linestyle="--", color="#d45500")
ax1.set_ylabel(ylabel[1])
ax1.yaxis.grid(True)
ax1.xaxis.grid(False)

# Third subplot with symmetric vertical error bars (Inflation Rate)
ax2.errorbar(
    x, inflation_rate, yerr=error_inflation_rate, fmt="*", color=color_inflation_rate, capsize=2
)
for i in range(len(x)):
    ax2.text(
        x[i],
        inflation_rate[i] + 0.2,
        f"{inflation_rate[i]:.2f}",
        fontsize=8,
        ha="center",
    )
ax2.set_ylabel(ylabel[2])
ax2.yaxis.grid(True)
ax2.xaxis.grid(False)

# Set shared x-axis labels
plt.xticks(x, months)
ax2.set_xlabel("Month")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("errorpoint_79.pdf", bbox_inches="tight")

