# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # For reproducibility

# Generate data using a preferential attachment model to simulate species interactions
G = nx.barabasi_albert_graph(30, 2)

# Position the nodes using the Kamada-Kawai layout
pos = nx.kamada_kawai_layout(G)

# Randomly select some edges to color green to signify significant ecological relationships
edges = list(G.edges())
green_edges = np.random.choice(len(edges), size=int(len(edges) * 0.3), replace=False)
green_edges = [edges[i] for i in green_edges]

# Labels and other descriptive parameters
xlabel = "Species Network Visualization"
ylabel = "Interaction Intensity"
title = "Ecosystem Food Web"
legend_labels = {"nodes": "Species", "edges": "Predator-Prey Relationships"}
label = "Highlighted Ecological Relationships"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))
ax.set_title(title)

# Draw the nodes with a specific color and size
nx.draw_networkx_nodes(G, pos, node_size=300, node_color="orange", ax=ax)

# Draw all edges with a base color
nx.draw_networkx_edges(G, pos, alpha=0.3, edge_color="#cccccc", ax=ax)

# Highlight selected edges in green
nx.draw_networkx_edges(G, pos, edgelist=green_edges, edge_color="green", width=2, ax=ax)

# Add labels
plt.xlabel(xlabel)
plt.ylabel(ylabel)

# Add a legend
from matplotlib.lines import Line2D

legend_elements = [
    Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        label=legend_labels["nodes"],
        markersize=10,
        markerfacecolor="orange",
    ),
    Line2D([0], [0], linestyle="-", color="green", label=label),
]

ax.legend(handles=legend_elements, loc="best")

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_26.pdf", bbox_inches="tight")
