# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective colors and types representing economic indicators
nodes = {
    0: {"color": "#FF6347", "type": "GDP"},
    1: {"color": "#4682B4", "type": "Inflation"},
    2: {"color": "#32CD32", "type": "Employment"},
    3: {"color": "#FFD700", "type": "Trade"},
    4: {"color": "#EE82EE", "type": "Investment"},
    5: {"color": "#8A2BE2", "type": "Consumer Confidence"},
    6: {"color": "#FF4500", "type": "Fiscal Policy"},
}
for node, attributes in nodes.items():
    G.add_node(node, color=attributes["color"], type=attributes["type"])

# Add edges with labels representing the influence pathways
edges = [
    (0, 2, "Economic Growth"),
    (1, 5, "Price Stability"),
    (2, 3, "Labor Market"),
    (3, 4, "Trade Balance"),
    (4, 0, "Investment Returns"),
    (5, 6, "Public Opinion"),
    (6, 1, "Policy Effect"),
]
for u, v, label in edges:
    G.add_edge(u, v, label=label)

# Define node positions in a circular layout
pos = nx.circular_layout(G)

title = "Economic Indicators Network"

legend_title = "Economic Indicators"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(
    G, pos, node_color=node_colors, node_size=700, edgecolors="black"
)

# Draw edges with dashed lines and arrowstyle
nx.draw_networkx_edges(
    G,
    pos,
    arrows=True,
    arrowstyle="-|>",
    style="dashed",
    connectionstyle="arc3,rad=0.2",
)

# Add edge labels
edge_labels = {(u, v): G[u][v]["label"] for u, v in G.edges}
nx.draw_networkx_edge_labels(G, pos, edge_labels=edge_labels, font_color="grey")

# Add node labels
nx.draw_networkx_labels(G, pos, {node: node for node in G.nodes}, font_color="black")

# Add a title
plt.title(title, fontsize=15)

# Create a legend for node types
legend_handles = [
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor=color,
        label=type,
        markersize=10,
        markeredgecolor="black",
    )
    for color, type in set(
        (attributes["color"], attributes["type"]) for attributes in nodes.values()
    )
]
ax.legend(
    handles=legend_handles, loc="upper right", fontsize="small", title=legend_title
)

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_42.pdf", bbox_inches="tight")
