# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
import numpy as np

# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective colors representing psychological service types
nodes = {
    0: {"color": "royalblue", "type": "Therapy"},
    1: {"color": "mediumseagreen", "type": "Counseling"},
    2: {"color": "gold", "type": "Psychiatry"},
    3: {"color": "crimson", "type": "Support Group"},
    4: {"color": "darkorchid", "type": "Therapy"},
    5: {"color": "lightsalmon", "type": "Counseling"},
    6: {"color": "dodgerblue", "type": "Psychiatry"},
}
for node, attributes in nodes.items():
    G.add_node(node, color=attributes["color"], type=attributes["type"])

# Add edges with labels representing types of interactions
edges = [
    (0, 1, "Referral"),
    (1, 2, "Collaboration"),
    (2, 3, "Referral"),
    (3, 4, "Collaboration"),
    (4, 5, "Supervision"),
    (5, 6, "Referral"),
]
for u, v, label in edges:
    G.add_edge(u, v, label=label)

# Define node positions in a circular layout
pos = nx.circular_layout(G)

# Text labels
title = "Psychological Services Interaction Network"

legend_title = "Service Type"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(
    G, pos, node_color=node_colors, node_size=700, edgecolors="black"
)

# Draw edges with labels and different styles
nx.draw_networkx_edges(
    G,
    pos,
    arrows=True,
    connectionstyle="arc3,rad=0.2",
    edge_color="gray",
    arrowstyle="-|>",
    arrowsize=15,
    style="dashed",
)
edge_labels = {(u, v): G[u][v]["label"] for u, v in G.edges}
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=edge_labels, font_color="red", font_size=9
)

# Add node labels
nx.draw_networkx_labels(
    G, pos, {node: node for node in G.nodes}, font_size=12, font_color="black"
)

# Add a title
plt.title(title, fontsize=15)

# Create a legend for node types
legend_handles = [
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor=color,
        label=type,
        markersize=10,
        markeredgecolor="black",
    )
    for color, type in set(
        (attributes["color"], attributes["type"]) for attributes in nodes.values()
    )
]
ax.legend(handles=legend_handles, loc="upper right", title=legend_title)

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_43.pdf", bbox_inches="tight")
