

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Ensuring reproducibility
# Create a random graph to represent a sports event network
G = nx.random_geometric_graph(50, 0.2)

# Position nodes using the Kamada-Kawai layout to achieve a more spread-out layout
pos = nx.kamada_kawai_layout(G)

# Node type assignment: 50% forwards (green), 50% defenders (blue)
node_colors = []
for node in G.nodes():
    if np.random.rand() > 0.5:
        node_colors.append("green")  # Forward
    else:
        node_colors.append("blue")  # Defender

# Edges: Mark 30% of the edges as experiencing disruptions (dotted lines, highlighted)
edges = list(G.edges())
num_disruptions = int(len(edges) * 0.3)
disruption_edges = np.random.choice(len(edges), size=num_disruptions, replace=False)
disruption_edges = [edges[i] for i in disruption_edges]

# Descriptive labels and title
title = "Player Interaction Network"
labels = ["Defender", "Forward", "Normal Interaction", "Disrupted Interaction"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw the nodes with respective colors
nx.draw_networkx_nodes(G, pos, node_size=100, node_color=node_colors, ax=ax)

# Draw all edges
nx.draw_networkx_edges(G, pos, ax=ax, edge_color="gray", alpha=0.5)

# Draw disruption edges with a different style (dotted)
nx.draw_networkx_edges(
    G, pos, ax=ax, edgelist=disruption_edges, style="dotted", edge_color="red"
)

# Add a legend
legend_elements = [
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor="blue",
        markersize=10,
        label=labels[0],
    ),
    plt.Line2D(
        [0],
        [0],
        marker="o",
        color="w",
        markerfacecolor="green",
        markersize=10,
        label=labels[1],
    ),
    plt.Line2D([0], [0], color="gray", lw=2, alpha=0.5, label=labels[2]),
    plt.Line2D([0], [0], color="red", linestyle=":", linewidth=2, label=labels[3]),
]

ax.legend(handles=legend_elements, loc="upper right")

# Remove axis
plt.axis("off")

# Add a title
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_45.pdf", bbox_inches="tight")

