
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import networkx as nx

np.random.seed(42)
# Create a random graph representing a soccer passing network
G = nx.random_geometric_graph(22, 0.5)

# Position the nodes using the Kamada-Kawai layout algorithm for a spread-out structure
pos = nx.kamada_kawai_layout(G)

# Randomly select some edges to color as 'key passes'
edges = list(G.edges())
key_passes = np.random.choice(
    len(edges), size=int(len(edges) * 0.2), replace=False
)
key_passes = [edges[i] for i in key_passes]

title = "Soccer Passing Network"
labels = ["Key Passes", "Regular Passes"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw the nodes with a gradient color
node_colors = np.linspace(0.4, 1.0, len(G.nodes()))
nx.draw_networkx_nodes(
    G, pos, node_size=300, node_color=node_colors, cmap=plt.cm.Blues, edgecolors="#1f3a93", linewidths=0.7
)

# Draw the edges with a light transparency
nx.draw_networkx_edges(G, pos, edge_color="#87CEEB", alpha=0.6)

# Draw the selected 'key passes' edges with a distinct color
nx.draw_networkx_edges(
    G, pos, edgelist=key_passes, edge_color="#800080", width=2.5
)

# Add a title
plt.title(title, fontsize=18)

# Add a legend
key_pass_patch = plt.Line2D(
    [], [], color="#800080", linewidth=2.5, linestyle="-", label=labels[0]
)
regular_pass_patch = plt.Line2D(
    [], [], color="#87CEEB", linewidth=2, linestyle="-", alpha=0.6, label=labels[1]
)
plt.legend(handles=[key_pass_patch, regular_pass_patch], fontsize=12)

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_48.pdf", bbox_inches="tight")
