
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(1)
# Creating a random graph representing healthcare providers and their patient connections
G = nx.random_geometric_graph(25, 0.3)

# Position the nodes using the Kamada-Kawai layout algorithm for a spread-out structure
pos = nx.kamada_kawai_layout(G)

# Randomly select some edges to color as 'primary care connections'
edges = list(G.edges())
primary_care_connections = np.random.choice(
    len(edges), size=int(len(edges) * 0.3), replace=False
)  # Highlighting 30% of the edges
primary_care_connections = [edges[i] for i in primary_care_connections]
title = "Healthcare Providers and Patient Connections Network"
labels = ["Primary Care Connections", "General Connections"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig = plt.subplots(figsize=(10, 10))

# Draw the nodes with a soft, thought-invoking color
node_color = "#87CEEB"
edgecolor = "#005073"
edge_light_color = "#98D8D8"
edge_primary_color = "#0B3D91"
title_color = "#003366"

nx.draw_networkx_nodes(
    G, pos, node_size=300, node_color=node_color, edgecolors=edgecolor, linewidths=0.5
)

# Draw the edges with a light transparency
nx.draw_networkx_edges(G, pos, edge_color=edge_light_color, alpha=0.5)

# Draw the selected 'primary care connections' edges with a distinct color
nx.draw_networkx_edges(
    G, pos, edgelist=primary_care_connections, edge_color=edge_primary_color, width=2
)

# Add a title
plt.title(title, fontsize=16, color=title_color)

# Add a legend
primary_patch = plt.Line2D(
    [], [], color=edge_primary_color, linewidth=2, linestyle="-", label=labels[0]
)
general_patch = plt.Line2D(
    [], [], color=edge_light_color, linewidth=2, linestyle="-", alpha=0.5, label=labels[1]
)
plt.legend(handles=[primary_patch, general_patch])

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("graph_50.pdf", bbox_inches="tight")
