
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
import numpy as np

# Create a directed graph representing a population migration network
G = nx.DiGraph()

# Add nodes representing different regions
regions = ['North', 'South', 'East', 'West', 'Central']
G.add_nodes_from(regions)

# Add edges representing migration flows between regions
migration_flows = [('North', 'South'), ('North', 'East'), ('South', 'West'), ('East', 'Central'), ('West', 'Central')]

# Add self-loops representing internal population activities
internal_activities = [('North', 'North'), ('South', 'South'), ('East', 'East'), ('West', 'West'), ('Central', 'Central')]

# Combine all edges
G.add_edges_from(migration_flows + internal_activities)

# Position nodes using a circular layout
pos = nx.circular_layout(G)

# Titles and Annotations
title = "Population Migration Network"
annotation_text = "Solid Lines: Migration Flows\nDashed Lines: Internal Activities"
xlabel = "X-axis: Network Layout"
ylabel = "Y-axis: Network Layout"
legend_labels = ["Migration Flows", "Internal Activities"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 8))

# Draw the graph with custom node and edge colors
node_color = "seagreen"
edge_color = "gray"
nx.draw(G, pos, with_labels=True, node_color=node_color, edge_color=edge_color, node_size=700)

# Highlight the migration flow edges with solid lines
migration_edge_color = "royalblue"
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=migration_flows,
    edge_color=migration_edge_color,
    style="solid",
    width=2.5,
    arrowstyle="-|>",
)

# Highlight the internal activities with dashed lines
internal_edge_color = "goldenrod"
nx.draw_networkx_edges(
    G, pos, edgelist=internal_activities, edge_color=internal_edge_color, style="dashed", width=2, arrowstyle="-|>"
)

# Add a title and annotations to the plot
plt.title(title, fontsize=16, fontweight='bold')
props = dict(boxstyle="round", facecolor="lightyellow", alpha=0.6)
plt.text(
    0.05,
    0.95,
    annotation_text,
    transform=plt.gca().transAxes,
    fontsize=12,
    verticalalignment="top",
    bbox=props,
)

# Axis labels
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_54.pdf", bbox_inches="tight")
