

# ===================
# Part 1: Importing Libraries
# ===================
import networkx as nx
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Create a directed graph for political entities
G = nx.DiGraph()
# Adding nodes representing political entities
G.add_nodes_from(
    [
        ("Executive", {"type": "entity"}),
        ("Legislative", {"type": "entity"}),
        ("Judicial", {"type": "entity"}),
        ("Media", {"type": "entity"}),
    ]
)

# Adding edges representing interactions
G.add_edges_from(
    [
        ("Executive", "Legislative"),
        ("Legislative", "Judicial"),
        ("Judicial", "Media"),
        ("Media", "Executive"),
    ]
)

# Adding self-loops representing internal processes
self_loops = [
    ("Executive", "Executive"),
    ("Legislative", "Legislative"),
    ("Judicial", "Judicial"),
    ("Media", "Media"),
]
G.add_edges_from(self_loops)

# Positioning nodes in a circular layout
pos = nx.circular_layout(G)

labels = ["Inter-entity Interactions", "Internal Entity Processes"]

title = "Interactions Between Political Entities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 8))

# Draw nodes and edges with base settings
color_map = {
    "Executive": "skyblue",
    "Legislative": "deepskyblue",
    "Judicial": "seagreen",
    "Media": "sienna",
}
node_colors = [color_map[node] for node in G.nodes()]

nx.draw(
    G,
    pos,
    with_labels=True,
    node_color=node_colors,
    node_size=3000,
    edge_color="lightgray",
    linewidths=1.5,
    font_size=12,
    font_weight="bold",
)

# Highlight self-loops with a different style
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=self_loops,
    edge_color="forestgreen",
    style="dashed",
    arrowstyle="<|-",
    arrowsize=20,
    width=2,
)

# Adding title and legend
plt.title(title, fontsize=16)
edge_legend = plt.Line2D([], [], linestyle="solid", color="lightgray", label=labels[0])
self_loop_legend = plt.Line2D(
    [], [], linestyle="dashed", color="forestgreen", label=labels[1]
)
plt.legend(handles=[edge_legend, self_loop_legend])

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_58.pdf", bbox_inches="tight")
