
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Create a directed graph
G = nx.DiGraph()

# Add nodes representing different entities in the logistics network
nodes = [
    "Port",
    "Warehouse",
    "Distribution Center",
    "Retail Store",
    "Customer",
]
for node in nodes:
    G.add_node(node)

# Define edges representing goods flow between entities
edges = [
    ("Port", "Warehouse"),
    ("Warehouse", "Distribution Center"),
    ("Distribution Center", "Retail Store"),
    ("Retail Store", "Customer"),
    ("Warehouse", "Warehouse"),  # Self-loop for internal logistics
    ("Customer", "Customer"),  # Self-loop for returns or exchanges
    ("Port", "Port"),  # Self-loop for port activities or customs
]

# Add edges to the graph
G.add_edges_from(edges)

# Define positions for a clear layout
pos = nx.spring_layout(G, seed=42)

title = "Goods Flow in a Logistics Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(12, 10))

# Define a new color scheme for the nodes
node_colors = [
    "green" if node == "Port" else "blue" if node == "Warehouse" 
    else "yellow" if node == "Distribution Center" else "orange" if node == "Retail Store" 
    else "purple" for node in G.nodes()
]

# Draw the graph with customized node colors and labels
nx.draw(
    G,
    pos,
    with_labels=True,
    node_color=node_colors,
    node_size=3000,
    font_size=10,
    font_weight="bold",
    edge_color="gray",
)

# Draw edges with differentiated styles for self-loops
self_loops = [(u, v) for u, v in G.edges() if u == v]
other_edges = [(u, v) for u, v in G.edges() if u != v]

nx.draw_networkx_edges(
    G,
    pos,
    edgelist=other_edges,
    arrowstyle="-|>",
    arrowsize=20,
    width=2.0,
    edge_color="black",
)
nx.draw_networkx_edges(
    G,
    pos,
    edgelist=self_loops,
    arrowstyle="-|>",
    style="dashed",
    arrowsize=20,
    width=2.0,
    edge_color="red",
)

# Title and labels
plt.title(title, fontsize=15)
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_62.pdf", bbox_inches="tight")
