

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Import necessary library
import networkx as nx
import numpy as np

# Create a directed graph
G = nx.DiGraph()

# Define nodes with historical figure-related attributes
historical_figures = {
    0: ("Julius Caesar", "blue"),    # Roman
    1: ("Cleopatra", "green"),      # Egyptian
    2: ("Alexander the Great", "orange"), # Macedonian
    3: ("Hannibal", "brown"),       # Carthaginian
    4: ("Augustus", "purple"),      # Roman
    5: ("Leonidas", "red"),         # Spartan
    6: ("Darius the Great", "cyan"), # Persian
}

for node, (figure, color) in historical_figures.items():
    G.add_node(node, figure=figure, color=color)

# Add edges with labels representing historical interactions
interactions = [
    (0, 1, "Alliance"),
    (1, 2, "Conflict"),
    (2, 3, "Battle"),
    (3, 4, "Rivalry"),
    (4, 5, "Succession"),
    (5, 6, "Invasion"),
    (6, 0, "Diplomacy"),
]
for u, v, label in interactions:
    G.add_edge(u, v, label=label)

# Define node positions using a spring layout for better visualization
pos = nx.spring_layout(G)

title = "Historical Figures Interaction Network"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw nodes with color attribute
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=800, ax=ax)

# Draw edges with various styles
nx.draw_networkx_edges(
    G,
    pos,
    arrows=True,
    arrowstyle="-|>",
    arrowsize=20,
    width=2,
    edge_color="black",
    ax=ax
)
edge_labels = {(u, v): G[u][v]["label"] for u, v in G.edges}
nx.draw_networkx_edge_labels(
    G, pos, edge_labels=edge_labels, font_color="darkred", font_size=10, ax=ax
)

# Draw node labels indicating historical figure names
node_labels = {node: G.nodes[node]["figure"] for node in G.nodes}
nx.draw_networkx_labels(
    G, pos, labels=node_labels, font_size=12, font_color="black", ax=ax
)

# Remove axis and add title
plt.axis("off")
plt.title(title, fontsize=16, fontweight="bold")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_64.pdf", bbox_inches="tight")

