
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx
import numpy as np

# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective types and colors
nodes = {
    0: ("Oak Tree", "deepskyblue"),
    1: ("Pine Tree", "mediumseagreen"),
    2: ("Mushroom Patch", "forestgreen"),
    3: ("Deer", "gold"),
    4: ("Rabbit", "lightcoral"),
    5: ("Wolf", "darkorange"),
    6: ("Bird", "limegreen"),
}
for node, (species, color) in nodes.items():
    G.add_node(node, species=species, color=color)

# Add edges with interaction types representing relationships in the ecosystem
edges = [
    (0, 2, "Symbiosis"),
    (1, 2, "Symbiosis"),
    (2, 3, "Food Source"),
    (2, 4, "Food Source"),
    (3, 5, "Predation"),
    (4, 5, "Predation"),
    (1, 6, "Shelter"),
    (0, 6, "Shelter"),
]
for u, v, interaction in edges:
    G.add_edge(u, v, interaction=interaction)

# Set network title and legend
title = "Forest Ecosystem Network"
legendtitle = "Ecosystem Components"

# Define node positions in a circular layout
pos = nx.circular_layout(G)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute and label them
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=700, node_shape='o')
nx.draw_networkx_labels(
    G, pos, labels={node: G.nodes[node]["species"] for node in G.nodes}, font_size=10
)

# Draw edges with interaction labels
nx.draw_networkx_edges(G, pos, arrowstyle="->", arrowsize=20, edge_color="dodgerblue")
edge_labels = {(u, v): G[u][v]["interaction"] for u, v in G.edges}
nx.draw_networkx_edge_labels(G, pos, edge_labels=edge_labels, font_size=8)

# Add a title
plt.title(title, fontsize=18)

# Add legend manually
legend_labels = {color: species for _, (species, color) in nodes.items()}
for color in set(node_colors):
    ax.plot(
        [],
        [],
        color=color,
        label=legend_labels[color],
        marker="o",
        markersize=10,
        linestyle="",
    )

# Removing the axis
plt.axis("off")

# Adding legend to the plot
plt.legend(title=legendtitle, loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_69.pdf", bbox_inches="tight")
