

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective court names and colors
nodes = {
    0: ("Court A", "forestgreen"),
    1: ("Court B", "darkorange"),
    2: ("Court C", "royalblue"),
    3: ("Office A1", "limegreen"),
    4: ("Office A2", "yellowgreen"),
    5: ("Office B1", "sandybrown"),
    6: ("Office C1", "skyblue"),
    7: ("Office C2", "cornflowerblue"),
}
for node, (location, color) in nodes.items():
    G.add_node(node, location=location, color=color)

# Add edges with durations representing legal proceedings
edges = [
    (0, 1, "12 months"),
    (0, 2, "18 months"),
    (1, 3, "2 months"),
    (1, 4, "5 months"),
    (2, 5, "9 months"),
    (2, 6, "7 months"),
    (2, 7, "10 months"),
]
for u, v, duration in edges:
    G.add_edge(u, v, duration=duration)

# Set network title and legend
title = "Legal Case Network Among Courts"
legendtitle = "Court and Office Nodes"

# Define node positions in a spring layout
pos = nx.spring_layout(G, seed=42)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 12))

# Draw nodes with color attribute and label them
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=800)
nx.draw_networkx_labels(
    G, pos, labels={node: G.nodes[node]["location"] for node in G.nodes}, font_size=10
)

# Draw edges with duration labels
nx.draw_networkx_edges(G, pos, edge_color='gray', arrowstyle="->", arrowsize=15, width=2)
edge_labels = {(u, v): G[u][v]["duration"] for u, v in G.edges}
nx.draw_networkx_edge_labels(G, pos, edge_labels=edge_labels, font_size=8)

# Add a title
plt.title(title, fontsize=20, fontweight='bold')

# Add legend manually
legend_labels = {color: location for _, (location, color) in nodes.items()}
for color in set(node_colors):
    ax.plot(
        [],
        [],
        color=color,
        label=legend_labels[color],
        marker="o",
        markersize=10,
        linestyle="",
    )

# Removing the axis
plt.axis("off")

# Adding legend to the plot
plt.legend(title=legendtitle, loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_70.pdf", bbox_inches="tight")

