

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import networkx as nx

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np  # Needed for generating data

# Create a directed graph
G = nx.DiGraph()

# Add nodes with their respective philosophical schools and colors
nodes = {
    0: ("Platonism", "darkblue"),
    1: ("Aristotelianism", "lightblue"),
    2: ("Stoicism", "lightgreen"),
    3: ("Skepticism", "orange"),
    4: ("Epicureanism", "pink"),
    5: ("Neoplatonism", "purple"),
    6: ("Existentialism", "red"),
}
for node, (school, color) in nodes.items():
    G.add_node(node, school=school, color=color)

# Add edges with descriptions of influence
edges = [
    (0, 1, "Aristotle was a student of Plato"),
    (1, 3, "Criticism of Aristotelian logic"),
    (2, 4, "Stoic ethics vs Epicurean ethics"),
    (0, 5, "Plotinus revived Platonism"),
    (5, 6, "Influence on Existentialism"),
    (1, 2, "Influence on Stoic logic"),
    (2, 3, "Stoic rejection of certainty"),
]
for u, v, influence in edges:
    G.add_edge(u, v, influence=influence)

# Set network title and legend
title = "Influence of Philosophical Schools"
legendtitle = "Philosophical Schools"

# Define node positions in a circular layout
pos = nx.circular_layout(G)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute and label them
node_colors = [G.nodes[node]["color"] for node in G.nodes]
nx.draw_networkx_nodes(G, pos, node_color=node_colors, node_size=700)
nx.draw_networkx_labels(
    G, pos, labels={node: G.nodes[node]["school"] for node in G.nodes}, font_size=10
)

# Draw edges with influence labels
nx.draw_networkx_edges(G, pos, arrowstyle="-|>", arrowsize=20)
edge_labels = {(u, v): G[u][v]["influence"] for u, v in G.edges}
nx.draw_networkx_edge_labels(G, pos, edge_labels=edge_labels, font_size=8)

# Add a title
plt.title(title, fontsize=18)

# Add legend manually
legend_labels = {color: school for _, (school, color) in nodes.items()}
for color in set(node_colors):
    ax.plot(
        [],
        [],
        color=color,
        label=legend_labels[color],
        marker="o",
        markersize=10,
        linestyle="",
    )

# Removing the axis
plt.axis("off")

# Adding legend to the plot
plt.legend(title=legendtitle, loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_71.pdf", bbox_inches="tight")
