
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Define regions and weather stations
regions = ["North", "South", "East", "West", "Central", "Northeast", "Northwest"]
weather_stations = [
    "StationA",
    "StationB",
    "StationC",
    "StationD",
    "StationE",
    "StationF",
    "StationG",
]

# Simulate annual average temperatures (in degrees Celsius)
temperature_data = np.array(
    [
        [5.2, 10.4, 15.6, 11.3, 7.5, 16.0, 9.0],
        [12.0, 9.1, 18.5, 6.9, 14.8, 11.1, 7.0],
        [8.5, 13.2, 10.0, 20.7, 10.6, 13.4, 12.1],
        [9.1, 11.9, 7.9, 10.4, 15.1, 10.9, 9.8],
        [5.5, 10.0, 13.1, 21.5, 11.2, 14.5, 7.2],
        [10.3, 8.7, 12.5, 14.9, 11.5, 16.9, 12.1],
        [6.6, 12.0, 10.3, 8.3, 13.9, 9.7, 17.2],
    ]
)

xlabel = "Regions"
ylabel = "Weather Stations"
title = "Annual Average Temperatures by Region and Weather Station"
cbarlabel = "Temperature (°C)"
output_filename = "heatmap_101_0.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar


def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.2f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts


fig, ax = plt.subplots(figsize=(10, 8))
im, cbar = heatmap(
    temperature_data, weather_stations, regions, ax=ax, cmap="coolwarm", cbarlabel=cbarlabel
)
texts = annotate_heatmap(im, valfmt="{x:.1f}°C")

# Title and labels
ax.set_title(title, fontsize=16, weight='bold')
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
fig.tight_layout()
plt.savefig("heatmap_101.pdf", bbox_inches="tight")

