
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

regions = ["North", "South", "East", "West", "Central", "Northeast", "Southwest"]
months = ["January", "February", "March", "April", "May", "June", "July"]

population_growth = np.array(
    [
        [1020, 1045, 1090, 1125, 1170, 1205, 1250],
        [980, 1025, 1060, 1102, 1137, 1173, 1210],
        [950, 998, 1025, 1070, 1105, 1138, 1175],
        [890, 920, 970, 1005, 1045, 1090, 1130],
        [1000, 1040, 1070, 1115, 1160, 1208, 1255],
        [1030, 1070, 1110, 1155, 1202, 1245, 1290],
        [920, 960, 1005, 1050, 1095, 1142, 1185],
    ]
)

# Setting the labels and titles
xlabel = "Months"
ylabel = "Regions"
title = "Monthly Population Growth in Different Regions"
cbarlabel = "Population Growth"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar


def annotate_heatmap(im, data=None, valfmt="{x:.2f}", textcolors=("black", "white"), threshold=None, **textkw):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts


fig, ax = plt.subplots(figsize=(10, 8))
im, cbar = heatmap(population_growth, regions, months, ax=ax, cmap="coolwarm", cbarlabel=cbarlabel)
texts = annotate_heatmap(im, valfmt="{x:.0f}")

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# ===================
# Part 4: Saving Output
# ===================
fig.tight_layout()
plt.savefig("heatmap_102.pdf", bbox_inches="tight")
