
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
genres = ["Fantasy", "Mystery", "Sci-Fi", "Romance", "Thriller", "Non-Fiction", "Historical"]
authors = [
    "Author A",
    "Author B",
    "Author C",
    "Author D",
    "Author E",
    "Author F",
    "Author G",
]

books_rated = np.array(
    [
        [4.5, 3.7, 4.8, 3.5, 4.0, 4.2, 3.8],
        [4.0, 3.5, 4.2, 4.8, 4.5, 3.7, 3.3],
        [3.8, 4.5, 4.0, 3.9, 4.3, 3.7, 4.1],
        [4.2, 4.7, 3.5, 4.1, 3.6, 4.3, 4.0],
        [3.9, 4.0, 4.3, 4.7, 3.4, 4.1, 3.8],
        [4.1, 3.8, 3.6, 4.3, 4.5, 3.9, 4.2],
        [4.3, 4.1, 3.7, 4.5, 4.2, 3.6, 3.9],
    ]
)

cbarlabels = ["average rating", "monthly books read", "correlation coeff."]
suptitle="Literature Analytics"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar

def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.2f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw,
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts

fig, ((ax, ax2), (ax3, ax4)) = plt.subplots(2, 2, figsize=(10, 8))

# Heatmap 1: Average ratings by genre and author
im, _ = heatmap(
    books_rated, genres, authors, ax=ax, cmap="Blues", cbarlabel=cbarlabels[0]
)
annotate_heatmap(im, valfmt="{x:.1f}", size=7)

# Heatmap 2: Random monthly books read data
data = np.random.randint(10, 50, size=(7, 7))
y = [f"Reader {i}" for i in range(1, 8)]
x = [f"Month {i}" for i in list("ABCDEFG")]
im, _ = heatmap(data, y, x, ax=ax2, vmin=10, cmap="Greens", cbarlabel=cbarlabels[1])
annotate_heatmap(im, valfmt="{x:d}", size=7, threshold=25, textcolors=("blue", "white"))

# Heatmap 3: Random data categorized using BoundaryNorm
data = np.random.randn(6, 6)
y = [f"Cat. {i}" for i in range(1, 7)]
x = [f"Year {i}" for i in range(1, 7)]

qrates = list("ABCDEFG")
norm = matplotlib.colors.BoundaryNorm(np.linspace(-3.5, 3.5, 8), 7)
fmt = matplotlib.ticker.FuncFormatter(lambda x, pos: qrates[::-1][norm(x)])

im, _ = heatmap(
    data,
    y,
    x,
    ax=ax3,
    cmap=mpl.colormaps["coolwarm"].resampled(7),
    norm=norm,
    cbar_kw=dict(ticks=np.arange(-3, 4), format=fmt),
    cbarlabel="Quality Rating",
)

annotate_heatmap(
    im, valfmt=fmt, size=9, fontweight="bold", threshold=-1, textcolors=("blue", "black")
)

# Heatmap 4: Correlation matrix of average book ratings
corr_matrix = np.corrcoef(books_rated)
im, _ = heatmap(
    corr_matrix,
    genres,
    genres,
    ax=ax4,
    cmap="bwr",
    vmin=-1,
    vmax=1,
    cbarlabel=cbarlabels[2],
)

def func(x, pos):
    return f"{x:.2f}".replace("0.", ".").replace("1.00", "")

annotate_heatmap(im, valfmt=matplotlib.ticker.FuncFormatter(func), size=7)

# ===================
# Part 4: Saving Output
# ===================
plt.suptitle(suptitle)
plt.tight_layout()
plt.savefig("heatmap_104.pdf", bbox_inches="tight")
