

# ===================
# Part 1: Importing Libraries
# ===================

import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Sample data: a 5x6 grid, representing interactions between geographical features
data = np.array(
    [
        [12.5, 35.2, 47.6, 28.0, 65.4, 75.3],
        [45.1, 30.3, 51.6, 10.1, 38.3, 59.7],
        [28.9, 50.6, 24.7, 42.3, 26.5, 74.2],
        [32.0, 55.1, 20.0, 15.3, 85.1, 66.6],
        [40.7, 48.9, 22.4, 58.5, 72.6, 63.0],
    ]
)

# X and Y labels
x_labels = ["Mountain", "Desert", "Forest", "River", "Ocean", "Plains"]
y_labels = ["Mountain", "Desert", "Forest", "River", "Ocean"]
xlabel = "Interacted Feature"
ylabel = "Interacting Feature"
title = "Interaction Heatmap Among Various Geographical Features"
cbar_label = "Interaction Percentage"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure and axis
fig, ax = plt.subplots(figsize=(10, 8))

# Defining the colormap from gold to deep purple
cmap = plt.cm.plasma

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Create the scatter plot
for i in range(data.shape[0]):
    for j in range(data.shape[1]):
        # Calculate the color based on the original value
        color = cmap(norm(data[i, j]))

        # Draw the circle with a fixed size
        circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
        ax.add_artist(circle)

        # Add the text inside the circle
        ax.text(j, i, f"{data[i, j]:.1f}%", ha="center", va="center", color="black")

# Set labels for x and y axes
ax.set_xticks(range(len(x_labels)))
ax.set_xticklabels(x_labels, ha="center")
ax.set_yticks(range(len(y_labels)))
ax.set_yticklabels(y_labels, va="center")

# Adding titles for the axes
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)
ax.set_title(title, fontsize=16)

# Set the limits of the axes; they should be one more than your data range
ax.set_xlim(-0.5, data.shape[1] - 0.5)
ax.set_ylim(-0.5, data.shape[0] - 0.5)

# Set the aspect of the plot to be equal and add a frame
ax.set_aspect("equal")
for spine in ax.spines.values():
    spine.set_visible(True)

# Create a colorbar
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = plt.colorbar(sm, ax=ax, ticks=[0.1, 1, 10, 100], orientation="vertical")
cbar.ax.set_yticklabels(["0.1", "1", "10", "100"])
cbar.set_label(cbar_label, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Save the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_59.pdf", bbox_inches="tight")

