
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import pandas as pd

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(77)

# Generating a similar style data for demonstration
# Assuming there are 3 seasons and 2 crops
seasons = ["Spring", "Summer", "Fall"]
crops = ["Crop A", "Crop B"]
agriculture_data = []

# Generating random yield productivity and soil quality rating for the example
for crop in crops:
    for season in seasons:
        soil_quality = np.random.randint(1, 10, size=9)  # Soil quality rating from 1 to 10
        yield_productivity = np.round(np.random.uniform(100, 500), 2)
        for rank, quality in enumerate(soil_quality, start=1):
            agriculture_data.append(
                {
                    "Crop": crop,
                    "Season": season,
                    "Rank": rank,
                    "SoilQuality": quality,
                    "YieldProductivity(kg)": yield_productivity,
                }
            )

# Create the DataFrame
df = pd.DataFrame(agriculture_data)

# Pivoting the dataframe for the heatmap
df_pivot = df.pivot_table(
    index="Rank", columns=["Crop", "Season"], values="SoilQuality", aggfunc="first"
)

# Chart label parameters
title = "Agriculture Soil Quality Heatmap"
xlabel = "Crop and Season"
ylabel = "Quality Rank"
color_bar_label = "Soil Quality"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Initialize the matplotlib figure
fig, ax = plt.subplots(figsize=(10, 10))

# Create a custom colormap for the heatmap
cmap = plt.get_cmap("viridis")

# Draw the heatmap
cax = ax.imshow(df_pivot, cmap=cmap)

# Decorations
plt.title(title, fontsize=14)
plt.xlabel(xlabel, fontsize=12)
plt.ylabel(ylabel, fontsize=15)

# Set x-axis and y-axis labels
ax.set_xticks(range(len(df_pivot.columns)))
ax.set_xticklabels(
    [f"{col[0]} {col[1]}" for col in df_pivot.columns], rotation=45, ha="right"
)
ax.set_yticks(range(len(df_pivot.index)))
ax.set_yticklabels(df_pivot.index)

# Add annotations
for i in range(len(df_pivot.index)):
    for j in range(len(df_pivot.columns)):
        ax.text(j, i, df_pivot.iloc[i, j], ha="center", va="center", color="white")

# Add colorbar to be smaller
cbar = fig.colorbar(cax, ax=ax, fraction=0.2, pad=0.04)
cbar.set_label(color_bar_label)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_77.pdf", bbox_inches="tight")
