
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import pandas as pd

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(78)

# Generating a similar style data for a new demonstration
# Assuming there are 3 health conditions and 3 hospitals
conditions = ["Cardiac Arrest", "Stroke", "Pneumonia"]
hospitals = ["Hospital A", "Hospital B", "Hospital C"]

admission_data = []

# Generating random admissions and percentage changes for the example
for condition in conditions:
    for hospital in hospitals:
        admissions = np.random.randint(20, 200, size=9)
        np.random.shuffle(admissions)
        percent_change = np.round(np.random.uniform(-50, 50), 2)
        for day in range(1, 10):
            admission_data.append(
                {
                    "Condition": condition,
                    "Hospital": hospital,
                    "Day": day,
                    "Admissions": admissions[day-1],
                    "PercentChange": percent_change,
                }
            )

# Create the DataFrame
df = pd.DataFrame(admission_data)

# Pivoting the dataframe for the heatmap
df_pivot = df.pivot_table(
    index="Day", columns=["Condition", "Hospital"], values="Admissions", aggfunc="first"
)

title = "Hospital Admissions Heatmap"
xlabel = "Condition and Hospital"
ylabel = "Day"
color_bar_label = "Number of Admissions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Initialize the matplotlib figure
fig, ax = plt.subplots(figsize=(10, 10))

# Create a custom colormap for the heatmap
cmap = plt.get_cmap("YlOrBr")

# Draw the heatmap
cax = ax.imshow(df_pivot, cmap=cmap)

# Decorations
plt.title(title, fontsize=16)
plt.xlabel(xlabel, fontsize=14)
plt.ylabel(ylabel, fontsize=14)

# Set x-axis and y-axis labels
ax.set_xticks(range(len(df_pivot.columns)))
ax.set_xticklabels(
    [f"{col[0]} - {col[1]}" for col in df_pivot.columns], rotation=45, ha="right"
)
ax.set_yticks(range(len(df_pivot.index)))
ax.set_yticklabels(df_pivot.index)

# Add annotations
for i in range(len(df_pivot.index)):
    for j in range(len(df_pivot.columns)):
        ax.text(j, i, df_pivot.iloc[i, j], ha="center", va="center", color="black")

# Add colorbar to be smaller
cbar = fig.colorbar(cax, ax=ax, fraction=0.02, pad=0.02)
cbar.set_label(color_bar_label)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("heatmap_78.pdf", bbox_inches="tight")
